<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Elementor mwt counter widget.
 *
 * Elementor widget that displays stats and numbers in an escalating manner.
 *
 * @since 1.0.0
 */
class Elementor_MWT_Counter_Up extends \Elementor\Widget_Base
{
    /**
     * Retrieve Widget Dependent JS.
     *
     * @since 1.0.0
     * @access public
     * @return array JS script handles.
     */
    public function get_script_depends()
    {
        return ['mwt-counter-up-script'];
    }

    /**
     * Retrieve Widget Dependent CSS.
     *
     * @since 1.0.0
     * @access public
     * @return array CSS style handles.
     */
    public function get_style_depends()
    {
        return ['mwt-counter-up-block-style'];
    }

    /**
     * Get widget name.
     *
     * Retrieve mwt counter widget name.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'mwt-counter-up';
    }

    /**
     * Get widget title.
     *
     * Retrieve counter widget title.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return esc_html__('MWT Counter Up', 'mwt-addons-for-elementor');
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the list widget belongs to.
     *
     * @since 1.0.0
     * @access public
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['mwt'];
    }

    /**
     * Get widget icon.
     *
     * Retrieve counter widget icon.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'eicon-counter';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @since 2.1.0
     * @access public
     *
     * @return array Widget keywords.
     */
    public function get_keywords()
    {
        return ['mwt', 'counter'];
    }

    /**
     * Register mwt counter widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 3.1.0
     * @access protected
     */
    protected function register_controls()
    {
        $this->start_controls_section(
            'section_counter',
            [
                'label' => esc_html__('Counter', 'mwt-addons-for-elementor'),
            ]
        );

        /* Start repeater */
        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'show_icon',
            [
                'label' => esc_html__('Show Icon', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'mwt-addons-for-elementor'),
                'label_off' => esc_html__('Hide', 'mwt-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $repeater->add_control(
            'counter_icon',
            [
                'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-circle',
                    'library' => 'fa-solid',
                ],
                'recommended' => [
                    'fa-solid' => [
                        'circle',
                        'dot-circle',
                        'square-full',
                    ],
                    'fa-regular' => [
                        'circle',
                        'dot-circle',
                        'square-full',
                    ],
                ],
                'condition' => [
                    'show_icon' => 'yes'
                ]
            ]
        );

        $repeater->add_control(
            'starting_number',
            [
                'label' => esc_html__('Starting Number', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 0,
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $repeater->add_control(
            'ending_number',
            [
                'label' => esc_html__('Ending Number', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 100,
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $repeater->add_control(
            'prefix',
            [
                'label' => esc_html__('Number Prefix', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => '',
                'placeholder' => 1,
            ]
        );

        $repeater->add_control(
            'suffix',
            [
                'label' => esc_html__('Number Suffix', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => '',
                'placeholder' => esc_html__('Plus', 'mwt-addons-for-elementor'),
            ]
        );

        $repeater->add_control(
            'title',
            [
                'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => esc_html__('Cool Number', 'mwt-addons-for-elementor'),
                'placeholder' => esc_html__('Cool Number', 'mwt-addons-for-elementor'),
            ]
        );

        $repeater->add_control(
            'title_bottom_line',
            [
                'label' => esc_html__('Title (bottom line)', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::TEXT,
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => '',
                'placeholder' => esc_html__('Cool Number', 'mwt-addons-for-elementor'),
            ]
        );

        $repeater->add_control(
            'counter_item_position',
            [
                'label' => esc_html__('Item Position', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'item-position-left' => [
                        'title' => esc_html__('Left', 'mwt-addons-for-elementor'),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'item-position-center' => [
                        'title' => esc_html__('Center', 'mwt-addons-for-elementor'),
                        'icon' => 'eicon-h-align-center',
                    ],
                    'item-position-right' => [
                        'title' => esc_html__('Right', 'mwt-addons-for-elementor'),
                        'icon' => 'eicon-h-align-right',
                    ],
                ],
                'default' => 'item-position-left',
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        /* End repeater */

        $this->add_control(
            'counter_list',
            [
                'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'show_icon' => 'yes',
                        'counter_icon' => '',
                        'starting_number' => 0,
                        'ending_number' => 100,
                        'prefix' => '',
                        'suffix' => '',
                        'title' => esc_html__('Title', 'mwt-addons-for-elementor'),
                        'title_bottom_line' => '',
                        'counter_item_position' => '',
                    ],
                ],
                'title_field' => '{{{ title }}}',
            ]
        );

        $this->add_control(
            'icon_options',
            [
                'label' => esc_html__('Icon Settings', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'counter_icon_position',
            [
                'label' => esc_html__('Icon Position', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::CHOOSE,
                'options' => [
                    'icon-position-left' => [
                        'title' => esc_html__('Left', 'mwt-addons-for-elementor'),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'icon-position-top' => [
                        'title' => esc_html__('Top', 'mwt-addons-for-elementor'),
                        'icon' => 'eicon-v-align-top',
                    ],
                    'icon-position-right' => [
                        'title' => esc_html__('Right', 'mwt-addons-for-elementor'),
                        'icon' => 'eicon-h-align-right',
                    ],
                ],
                'default' => 'icon-position-top',
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'separator_options',
            [
                'label' => esc_html__('Separator', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'thousand_separator',
            [
                'label' => esc_html__('Thousand Separator', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'default' => 'label_on',
                'label_on' => esc_html__('Show', 'mwt-addons-for-elementor'),
                'label_off' => esc_html__('Hide', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'thousand_separator_char',
            [
                'label' => esc_html__('Separator', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'condition' => [
                    'thousand_separator' => 'yes',
                ],
                'options' => [
                    '' => 'Default',
                    '.' => 'Dot',
                    ' ' => 'Space',
                    '_' => 'Underline',
                    "'" => 'Apostrophe',
                ],
            ]
        );

        $this->add_control(
            'duration',
            [
                'label' => esc_html__('Animation Duration', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'default' => 2000,
                'min' => 100,
                'step' => 100,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'view',
            [
                'label' => esc_html__('View', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HIDDEN,
                'default' => 'traditional',
            ]
        );

        $this->end_controls_section();

        //style
        $this->start_controls_section(
            'style_general_section',
            [
                'label' => esc_html__('General', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'items_display_property',
            [
                'label' => esc_html__('Display', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'block',
                'options' => [
                    'block' => esc_html__('Default', 'mwt-addons-for-elementor'),
                    'flex'  => esc_html__('Flex', 'mwt-addons-for-elementor'),
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-counters-grid-wrapper' => 'display: {{VALUE}};',
                ],
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_responsive_control(
            'items_justify_content',
            [
                'label' => esc_html__('Justify Content', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'start',
                'options' => [
                    'start'         => esc_html__('Start', 'mwt-addons-for-elementor'),
                    'end'           => esc_html__('End', 'mwt-addons-for-elementor'),
                    'center'        => esc_html__('Center', 'mwt-addons-for-elementor'),
                    'space-around'  => esc_html__('Space Around', 'mwt-addons-for-elementor'),
                    'space-between' => esc_html__('Space Between', 'mwt-addons-for-elementor'),
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-counters-grid-wrapper' => 'justify-content: {{VALUE}};',
                ],
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
                    'items_display_property' => 'flex',
                ]
            ]
        );

        $this->add_responsive_control(
            'items_flex_wrap',
            [
                'label' => esc_html__('Flex Wrap', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'nowrap',
                'options' => [
                    'nowrap'         => esc_html__('No', 'mwt-addons-for-elementor'),
                    'wrap'           => esc_html__('Wrap', 'mwt-addons-for-elementor'),
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-counters-grid-wrapper' => 'flex-wrap: {{VALUE}};',
                ],
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
                    'items_display_property' => 'flex',
                ]
            ]
        );

        $this->add_control(
            'item_style',
            [
                'label' => esc_html__('Item', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_responsive_control(
            'item_width',
            [
                'label'      => __('Width', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range'      => [
                    'px' => [
                        'min'  => 10,
                        'max'  => 1000,
                        'step' => 1,
                    ],
                    '%' => [
                        'min'  => 1,
                        'max'  => 100,
                        'step' => 1,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => 280,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-counter-up' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_responsive_control(
            'item_margin',
            [
                'label'      => __('Margin', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-counter-up' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'show_animation_pattern',
            [
                'label' => esc_html__('Show Animation Pattern', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
                'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'no',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'animation_pattern_color',
            [
                'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-counter-up-animation-pattern::before, .mwt-counter-up-animation-pattern::after, .mwt-counter-up-animation-pattern span::before, .mwt-counter-up-animation-pattern span::after' => 'background-color: {{VALUE}};',
                ],
                'condition' => [
                    'show_animation_pattern' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'animation_pattern_position_left',
            [
                'label'      => __('Left Position', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -100,
                        'max'  => 100,
                        'step' => 1,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => -15,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-counter-up-animation-pattern' => 'left: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'show_animation_pattern' => 'yes',
                ],
            ]
        );

        $this->add_responsive_control(
            'animation_pattern_position_top',
            [
                'label'      => __('Top Position', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'range'      => [
                    'px' => [
                        'min'  => -100,
                        'max'  => 100,
                        'step' => 1,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => -15,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-counter-up-animation-pattern' => 'top: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'show_animation_pattern' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'style_icon_section',
            [
                'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'counter_icon_size',
            [
                'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em'],
                'range'      => [
                    'px' => [
                        'min'  => 8,
                        'max'  => 120,
                        'step' => 1,
                    ],
                ],
                'default'    => [
                    'unit' => 'px',
                    'size' => 90,
                ],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-counter-up-icon i' => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .mwt-counter-up-icon svg' => 'max-width: {{SIZE}}{{UNIT}}; max-height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'icon_color',
            [
                'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-counter-up-icon i' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .mwt-counter-up-icon svg' => 'fill: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'icon_margin',
            [
                'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .mwt-counter-up-icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_number',
            [
                'label' => esc_html__('Number', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'number_position',
            [
                'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => '',
                'options' => [
                    '' => esc_html__('Default', 'mwt-addons-for-elementor'),
                    '2' => esc_html__('Bottom', 'mwt-addons-for-elementor'),
                ],
                'selectors' => [
                    '{{WRAPPER}} .mwt-counter-up-number-wrapper' => 'order: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'number_color',
            [
                'label' => esc_html__('Text Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-counter-up-number-wrapper' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'typography_number',
                'selector' => '{{WRAPPER}} .mwt-counter-up-number-wrapper',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Stroke::get_type(),
            [
                'name' => 'number_stroke',
                'selector' => '{{WRAPPER}} .mwt-counter-up-number-wrapper',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'number_shadow',
                'selector' => '{{WRAPPER}} .mwt-counter-up-number-wrapper',
            ]
        );

        $this->add_responsive_control(
            'number_bottom_spacing',
            [
                'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-counter-up-number-wrapper'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
                ],
                'condition' => [
                    'number_position!' => '2',
                ]
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'section_title',
            [
                'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'title_color',
            [
                'label' => esc_html__('Text Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-counter-up-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'typography_title',
                'selector' => '{{WRAPPER}} .mwt-counter-up-title',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Stroke::get_type(),
            [
                'name' => 'title_stroke',
                'selector' => '{{WRAPPER}} .mwt-counter-up-title',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'title_shadow',
                'selector' => '{{WRAPPER}} .mwt-counter-up-title',
            ]
        );

        $this->add_responsive_control(
            'title_bottom_spacing',
            [
                'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-counter-up-title'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
                ],
                'condition' => [
                    'number_position' => '2',
                ]
            ]
        );

        $this->add_control(
            'title_bottom_options',
            [
                'label' => esc_html__('Title (bottom line)', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'title_bottom_color',
            [
                'label' => esc_html__('Text Color', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .mwt-counter-up-title-bottom' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Typography::get_type(),
            [
                'name' => 'typography_title_bottom',
                'selector' => '{{WRAPPER}} .mwt-counter-up-title-bottom',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Stroke::get_type(),
            [
                'name' => 'title_bottom_stroke',
                'selector' => '{{WRAPPER}} .mwt-counter-up-title-bottom',
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Text_Shadow::get_type(),
            [
                'name' => 'title_bottom_shadow',
                'selector' => '{{WRAPPER}} .mwt-counter-up-title-bottom',
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render mwt counter widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display(); ?>
        <div class="mwt-counters-container">
            <div class="mwt-counters-grid-wrapper">
                <?php
                if (!empty($settings['thousand_separator'])) {
                    $delimiter = empty($settings['thousand_separator_char']) ? ',' : $settings['thousand_separator_char'];
                    $data_delimiter = 'data-delimiter="' . esc_attr($delimiter) . '"';
                }

                foreach ($settings['counter_list'] as $index => $item) { ?>
                    <div class="mwt-counter-up <?php echo esc_attr($settings['counter_icon_position'] . ' ' . $item['counter_item_position']) ?>">
                        <div class="mwt-counter-up-wrap">
                            <?php
                            $counter_icon = isset($item['counter_icon']);
                            if ('yes' === $item['show_icon'] && $counter_icon) : ?>
                                <div class="mwt-counter-up-icon">
                                    <?php \Elementor\Icons_Manager::render_icon($item['counter_icon'], ['aria-hidden' => 'true', 'class' => 'counter-icon']); ?>
                                </div>
                            <?php endif; ?>
                            <div class="mwt-counter-up-content-wrapper">
                                <?php if ('yes' === $settings['show_animation_pattern']) : ?>
                                    <div class="mwt-counter-up-animation-pattern">
                                        <span></span>
                                    </div>
                                <?php endif; ?>
                                <?php
                                $data_duration = $settings['duration'];
                                $data_to_value = $settings['counter_list'][$index]['ending_number'];
                                $data_from_value = $settings['counter_list'][$index]['starting_number'];
                                $prefix = $settings['counter_list'][$index]['prefix'];
                                $suffix = $settings['counter_list'][$index]['suffix'];
                                ?>
                                <div class="mwt-counter-up-number-wrapper">
                                    <span class="mwt-counter-up-number-prefix"><?php echo esc_html($prefix); ?></span>
                                    <span class="mwt-counter-up-number <?php echo esc_attr($index) ?>" data-duration="<?php echo esc_attr($data_duration); ?>" data-to-value="<?php echo esc_attr($data_to_value); ?>" data-from-value="<?php echo esc_attr($data_from_value); ?>" <?php echo (!empty($settings['thousand_separator'])) ? $data_delimiter : ''; ?>> <?php echo esc_html($data_from_value); ?></span>
                                    <span class="mwt-counter-up-number-suffix"><?php echo esc_html($suffix); ?></span>
                                </div>
                                <?php
                                $title = $settings['counter_list'][$index]['title'];
                                if ($title) : ?>
                                    <div class="mwt-counter-up-title"><?php echo esc_html($title); ?></div>
                                <?php endif;
                                $title_bottom = $settings['counter_list'][$index]['title_bottom_line'];
                                if ($title_bottom) : ?>
                                    <div class="mwt-counter-up-title-bottom"><?php echo esc_html($title_bottom); ?></div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                <?php } ?>
            </div>
        </div>
<?php
    }
}
