<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Gallery Carousel Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Gallery_Carousel extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent JS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array JS script handles.
	 */
	public function get_script_depends()
	{
		return ['mwt-gallery-carousel-script'];
	}

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['owl-carousel-style'];
		return ['mwt-gallery-carousel-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'mwt-gallery-carousel';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Gallery Carousel', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image carousel widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-slider-push';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'image', 'photo', 'visual', 'gallery', 'carousel', 'slider'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Carousel Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'items_layout',
			[
				'label' => esc_html__('Item Layout', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'label_block' => true,
				'default' => 'item-regular',
				'options' => [
					'item-regular' => esc_html__('Image with categories, title and description', 'mwt-addons-for-elementor'),
					'item-extended' => esc_html__('Image with categories and title over thumbnail', 'mwt-addons-for-elementor'),
					'item-image' => esc_html__('Image only', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		/* Start repeater */

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'item_image',
			[
				'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'dynamic' => [
					'active' => true,
				],
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$repeater->add_control(
			'item_title',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('This is the title', 'mwt-addons-for-elementor'),
				'default' => esc_html__('This is the title', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'item_title_link',
			[
				'label'       => __('Title Link', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::URL,
				'placeholder' => 'https://example.com/',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'show_link',
			[
				'label' => esc_html__('Show link icon over thumbnail', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'description' => esc_html__('Works only when layout set to \'Image with categories, title and description\' or \'Image only\' and \'Title Link\' is not empty', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
				'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$repeater->add_control(
			'item_category',
			[
				'label'       => __('Category', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Category', 'mwt-addons-for-elementor'),
				'description' => __('To assign for multiple categories, separate by a comma \',\'', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$repeater->add_control(
			'item_description',
			[
				'label'       => __('Description', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::TEXTAREA,
				'placeholder' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'description' => __('For Layout \'Image with categories, title and description\'', 'mwt-addons-for-elementor'),
				'default' => esc_html__('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'mwt-addons-for-elementor'),
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		/* End repeater */

		$this->add_control(
			'gallery-list',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'item_image' => '',
						'item_title' => esc_html__('Title 1', 'mwt-addons-for-elementor'),
						'item_title_link' => '',
						'show_link' => '',
						'item_category' => esc_html__('Category 1', 'mwt-addons-for-elementor'),
						'item_description' => esc_html__('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'mwt-addons-for-elementor'),
					],
					[
						'item_image' => '',
						'item_title' => esc_html__('Title 2', 'mwt-addons-for-elementor'),
						'item_title_link' => '',
						'show_link' => '',
						'item_category' => esc_html__('Category 2', 'mwt-addons-for-elementor'),
						'item_description' => esc_html__('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'mwt-addons-for-elementor'),
					],
					[
						'item_image' => '',
						'item_title' => esc_html__('Title 3', 'mwt-addons-for-elementor'),
						'item_title_link' => '',
						'show_link' => '',
						'item_category' => esc_html__('Category 3', 'mwt-addons-for-elementor'),
						'item_description' => esc_html__('Lorem ipsum dolor sit amet, consectetur adipiscing elit. Ut elit tellus, luctus nec ullamcorper mattis, pulvinar dapibus leo.', 'mwt-addons-for-elementor'),
					],
				],
				'title_field' => '{{{ item_title }}}',
			]
		);

		$this->add_control(
			'gallery_links_icon',
			[
				'label'     => __('Links Icon', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::ICONS,
				'default' => [
					'library' => 'fa-solid',
					'value'   => 'fas fa-link',
				],
				'recommended' => [
					'fa-solid' => [
						'link',
					],
				],
				'separator' => 'before',
				'condition' => [
					'items_layout' => ['item-regular', 'item-image'],
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_cats',
			[
				'label' => __('Categories', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter',
			[
				'label'   => __('Filter Tabs', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->add_control(
			'gallery_first_cat_label',
			[
				'label'     => __('First Category Label', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => __('All', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'gallery_filter' => 'yes',
				],
			]
		);

		$category_repeater = new \Elementor\REPEATER();

		$category_repeater->add_control(
			'gallery_cat',
			[
				'label'   => __('Category', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$category_repeater->add_control(
			'gallery_cat_link',
			[
				'label'       => __('Link', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::URL,
				'placeholder' => 'https://example.com/',
				'label_block' => true,
				'dynamic' => [
					'active' => true,
				],
			]
		);

		$this->add_control(
			'gallery_cats_content',
			[
				'label'       => __('Categories', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::REPEATER,
				'fields'      => $category_repeater->get_controls(),
				'default'     => [
					[
						'gallery_cat' => esc_html__('Category 1', 'mwt-addons-for-elementor'),
						'gallery_cat_link' => '',
					],
					[
						'gallery_cat' => esc_html__('Category 2', 'mwt-addons-for-elementor'),
						'gallery_cat_link' => '',
					],
					[
						'gallery_cat' => esc_html__('Category 3', 'mwt-addons-for-elementor'),
						'gallery_cat_link' => '',
					],
				],
				'title_field' => '{{{ gallery_cat }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_lightbox_section',
			[
				'label' => __('Lightbox', 'mwt-addons-for-elementor'),
				'condition' => [
					'items_layout' => ['item-regular', 'item-image'],
				],
			]
		);

		$this->add_control(
			'open_lightbox',
			[
				'label'   => __('Lightbox', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'description' => sprintf(
					esc_html__('Manage your site’s lightbox settings in the %1$sLightbox panel%2$s.', 'mwt-addons-for-elementor'),
					'<a href="javascript: $e.run( \'panel/global/open\' ).then( () => $e.route( \'panel/global/settings-lightbox\' ) )">',
					'</a>'
				),
				'default' => 'no',
			]
		);

		$this->add_control(
			'gallery_lightbox_icon',
			[
				'label'     => __('Lightbox Icon', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::ICONS,
				'default'   => [
					'library' => 'fa-solid',
					'value'   => 'fas fa-search',
				],
				'recommended' => [
					'fa-solid' => [
						'search',
					],
				],
				'condition' => [
					'open_lightbox' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_settings',
			[
				'label' => __('Carousel Settings', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'thumbnail',
				'separator' => 'none',
			]
		);

		$this->add_control(
			'data_loop',
			[
				'label'   => __('Loop', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_rtl',
			[
				'label'   => __('Enable RTL', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay',
			[
				'label'   => __('Autoplay', 'mwt-addons-for-elementor'),
				'description' => esc_html__('Slide will start automatically', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);


		$this->add_control(
			'data_autoplaytimeout',
			[
				'label' => esc_html__('Autoplay Speed (ms)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 5000,
				'min' => 0,
				'step' => 50,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay_hover_pause',
			[
				'label'   => __('Pause on Hover', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_center',
			[
				'label'   => __('Centered Carousel?', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
			]
		);

		//Slides to Scroll

		$this->add_control(
			'data_responsive_lg',
			[
				'label' => esc_html__('Desktop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 4,
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_md',
			[
				'label' => esc_html__('Laptop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_sm',
			[
				'label' => esc_html__('Tablet Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 2,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_xs',
			[
				'label' => esc_html__('Mobile Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 1,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_margin',
			[
				'label' => esc_html__('Item Gap (px)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 0,
				'max' => 80,
				'step' => 1,
				'default' => 30,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_nav_prev_next',
			[
				'label'   => __('Show prev/next', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'date_nav_prev',
			[
				'label'            => __('Previous Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-chevron-left',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'chevron-left',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'date_nav_prev_word',
			[
				'label'     => __('Previous word', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Prev', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'data_nav_prev_next' => 'yes',
				],
			]
		);

		$this->add_control(
			'date_nav_next',
			[
				'label'            => __('Next Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-chevron-right',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'chevron-right',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'date_nav_next_word',
			[
				'label'     => __('Next word', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Next', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'data_nav_prev_next' => 'yes',
				],
			]
		);

		$this->add_control(
			'data_nav_dots',
			[
				'label'   => __('Show dots', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->end_controls_section();

		//style
		$this->start_controls_section(
			'style_item_section',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'item_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'Left',
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-content-wrap' => 'text-align: {{VALUE}}',
				],
				'condition'   => [
					'items_layout!' => 'item-image',
				],
			]
		);

		$this->add_responsive_control(
			'item_content_padding',
			[
				'label'      => __('Content Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-content-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
				'condition'   => [
					'items_layout!' => 'item-image',
				],
			]
		);

		$this->add_control(
			'item_background_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-carousel-item' => 'background-color: {{VALUE}};',
				],
				'condition'   => [
					'items_layout!' => 'item-image',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'items_border',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-container .owl-carousel-item',
			]
		);

		$this->add_control(
			'items_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-carousel-item' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'items_style_shadow',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-container .owl-carousel-item',
			]
		);

		$this->add_responsive_control(
			'item_padding',
			[
				'label'      => __('Carousel Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-stage-outer-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'item_margin',
			[
				'label'      => __('Carousel Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-stage-outer-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_image',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'image_hover_overlay_color',
			[
				'label' => esc_html__('Hover Overlay Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-wrap::before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-container .layout-item-extended .mwt-gallery-carousel-content-wrap::before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .mwt-gallery-carousel-title-link::before' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'image_hover_overlay_opacity',
			[
				'label'      => __('Opacity', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					[
						'min' => 0.1,
						'max' => 1,
						'step' => 0.01,
					]
				],
				'default' => [
					'size' => 1,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap:hover .links-wrap::before'  => 'opacity: {{SIZE}}',
					'{{WRAPPER}} .mwt-gallery-carousel-container .layout-item-extended .mwt-gallery-carousel-content-wrap:hover::before' => 'opacity: {{SIZE}}',
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .mwt-gallery-carousel-title-link:hover::before' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_categories_section',
			[
				'label' => esc_html__('Categories', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'items_layout!' => 'item-image',
				],
			]
		);

		$this->add_control(
			'categories_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-categories' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'categories_color_hover',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} a.mwt-gallery-carousel-categories:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'categories_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-gallery-carousel-categories',
			]
		);

		$this->add_responsive_control(
			'categories_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-categories-wrap'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_title_section',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'items_layout!' => 'item-image',
				],
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-title' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_color_hover',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-title a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'title_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-gallery-carousel-title',
				'{{WRAPPER}} .mwt-gallery-carousel-title a',
			]
		);

		$this->add_responsive_control(
			'title_bottom_spacing',
			[
				'label'      => __('Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-title'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_description_section',
			[
				'label' => esc_html__('Description', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'items_layout!' => ['item-extended', 'item-image'],
				],
			]
		);

		$this->add_control(
			'description_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-description' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'description_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-gallery-carousel-description',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'icons_style',
			[
				'label' => __('Icons', 'mwt-addons-for-elementor'),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'items_layout' => ['item-regular', 'item-image'],
				],
			]
		);

		$this->add_responsive_control(
			'icons_size',
			[
				'label'       => __('Size', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::SLIDER,
				'size_units'  => ['px', 'em'],
				'range'       => [
					'px' => [
						'min' => 1,
						'max' => 100,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'label_block' => true,
				'selectors'   => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap' => 'font-size: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->start_controls_tabs('icons_style_tabs');

		$this->start_controls_tab(
			'icons_style_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'icons_style_color',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'icons_style_background',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'icons_style_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'icons_style_color_hover',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a:hover i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a:hover svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'icons_style_background_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'icons_style_border_color_hover',
			[
				'label'     => __('Border Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'icons_style_border',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a',
			]
		);

		$this->add_control(
			'icons_style_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
				'separator' => 'after',
			]
		);

		//lightbox icon style

		$this->add_control(
			'lightbox_icon',
			[
				'label' => esc_html__('lightbox Icon Style', 'textdomain'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('On', 'textdomain'),
				'label_off' => esc_html__('Off', 'textdomain'),
				'return_value' => 'yes',
				'default' => 'no',
				'condition' => [
					'open_lightbox' => 'yes',
				]
			]
		);

		$this->start_controls_tabs('lightbox_icons_style_tabs');

		$this->start_controls_tab(
			'lightbox_icons_style_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
				'condition' => [
					'lightbox_icon' => 'yes',
				]
			]
		);

		$this->add_control(
			'lightbox_icons_style_color',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a.mwt-gallery-carousel-lightbox-icon i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a.mwt-gallery-carousel-lightbox-icon svg' => 'fill: {{VALUE}};',
				],
				'condition' => [
					'lightbox_icon' => 'yes',
				]
			]
		);

		$this->add_control(
			'lightbox_icons_style_background',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a.mwt-gallery-carousel-lightbox-icon' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'lightbox_icon' => 'yes',
				]
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'lightbox_icons_style_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
				'condition' => [
					'lightbox_icon' => 'yes',
				]
			]
		);

		$this->add_control(
			'lightbox_icons_style_color_hover',
			[
				'label'     => __('Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a.mwt-gallery-carousel-lightbox-icon:hover i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a.mwt-gallery-carousel-lightbox-icon:hover svg' => 'fill: {{VALUE}};',
				],
				'condition' => [
					'lightbox_icon' => 'yes',
				]
			]
		);

		$this->add_control(
			'lightbox_icons_style_background_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a.mwt-gallery-carousel-lightbox-icon:hover' => 'background-color: {{VALUE}};',
				],
				'condition' => [
					'lightbox_icon' => 'yes',
				]
			]
		);

		$this->add_control(
			'lightbox_icons_style_border_color_hover',
			[
				'label'     => __('Border Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a.mwt-gallery-carousel-lightbox-icon:hover' => 'border-color: {{VALUE}};',
				],
				'condition' => [
					'lightbox_icon' => 'yes',
				]
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'lightbox_icons_style_border',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a.mwt-gallery-carousel-lightbox-icon',
				'condition' => [
					'lightbox_icon' => 'yes',
				]
			]
		);

		$this->add_control(
			'lightbox_icons_style_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a.mwt-gallery-carousel-lightbox-icon' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'lightbox_icon' => 'yes',
				],
				'separator' => 'after',
			]
		);

		//end lightbox icon style

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'label'    => __('Shadow', 'mwt-addons-for-elementor'),
				'name'     => 'icons_style_shadow',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a',
			]
		);

		$this->add_responsive_control(
			'icons_style_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'icons_style_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-image-wrap .links-inner-wrap a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'gallery_filter_style',
			[
				'label'     => __('Filter', 'mwt-addons-for-elementor'),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'gallery_filter' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_filters_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters' => 'text-align: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'gallery_filter_typo',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-container .filters a.category',
			]
		);

		$this->start_controls_tabs('gallery_filters');

		$this->start_controls_tab(
			'gallery_filters_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a.category' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a.category' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'gallery_filters_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter_hover_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_background_hover_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_border_hover_color',
			[
				'label'     => __('Border Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'gallery_filters_active',
			[
				'label' => __('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'gallery_filter_active_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a.active' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_background_active_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a.active' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'gallery_border_active_color',
			[
				'label'     => __('Border Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a.active' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'gallery_filter_border',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-container .filters a.category',
			]
		);

		$this->add_control(
			'gallery_filter_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a.category'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'gallery_filter_shadow',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-container .filters a.category',
			]
		);

		$this->add_responsive_control(
			'gallery_filter_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a.category' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_filter_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters a.category' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'gallery_filters_bottom_spacing',
			[
				'label'      => __('Filters Bottom Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .filters'  => 'margin-bottom: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_prev_next_section',
			[
				'label' => esc_html__('Prev/Next', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'prev_next_icons_position',
			[
				'label' => esc_html__('Prev/Next Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'bottom',
				'options' => [
					'bottom' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'middle' => esc_html__('Middle', 'mwt-addons-for-elementor'),
					'middle-left' => esc_html__('Middle Left', 'mwt-addons-for-elementor'),
				],
			]
		);

		$this->add_control(
			'prev_next_align',
			[
				'label' => esc_html__('Alignment', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__('Start', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-start-h',
					],
					'center' => [
						'title' => esc_html__('Center', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-center-h',
					],
					'end' => [
						'title' => esc_html__('End', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-end-h',
					],
					'space-between' => [
						'title' => esc_html__('Space Between', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-justify-space-between-h',
					],
				],
				'default' => 'start',
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav' => 'justify-content: {{VALUE}};',
				],
				'condition' => [
					'prev_next_icons_position!' => 'middle-left',
				]
			]
		);

		$this->add_responsive_control(
			'prev_next_offset',
			[
				'label'      => __('Offset', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range'      => [
					'px' => [
						'min'  => -1000,
						'max'  => 1000,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav' => 'margin-left: calc({{SIZE}}{{UNIT}} * -1); margin-right: calc({{SIZE}}{{UNIT}} * -1); width: calc(100% + {{SIZE}}{{UNIT}} * 2);',
				],
				'condition' => [
					'prev_next_icons_position!' => 'middle-left',
				]
			]
		);

		$this->add_responsive_control(
			'prev_next_offset_middle_left',
			[
				'label'      => __('Offset', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range'      => [
					'px' => [
						'min'  => -1000,
						'max'  => 1000,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav' => 'margin-left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'prev_next_icons_position' => 'middle-left',
				]
			]
		);

		$this->add_control(
			'hr',
			[
				'type' => \Elementor\Controls_Manager::DIVIDER,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'prev_next_typography',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button',
			]
		);

		$this->add_responsive_control(
			'prev_next_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 4,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button svg' => 'font-size: {{SIZE}}{{UNIT}};',

				],
			]
		);

		$this->add_control(
			'prev_next_icon_position',
			[
				'label' => esc_html__('Icon Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'row',
				'options' => [
					'row' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'row-reverse' => esc_html__('Reverse', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button>span' => 'flex-direction: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_icon_spacing',
			[
				'label'      => __('Icon Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 12,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button>span' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs(
			'prev_next_style_tabs'
		);

		$this->start_controls_tab(
			'prev_next_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color',
			[
				'label' => esc_html__('Text Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'prev_next_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color_hover',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button:hover i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button:hover svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'prev_next_style_disabled_tab',
			[
				'label' => esc_html__('Disabled', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color_disabled',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button.disabled' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button.disabled svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color_disabled',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button.disabled i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button.disabled svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color_disabled',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button.disabled' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_border_color_disabled',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button.disabled' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'prev_next_border',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button',
			]
		);

		$this->add_control(
			'prev_next_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_padding',
			[
				'label'      => esc_html__('Pdding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'allowed_dimensions' => 'vertical',
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav' => 'margin-top: {{TOP}}{{UNIT}}; margin-bottom: {{BOTTOM}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_btn_gap',
			[
				'label'      => __('Buttons Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-nav' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_dots_section',
			[
				'label' => esc_html__('Dots', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'data_nav_dots' => 'yes',
				]
			]
		);

		$this->start_controls_tabs(
			'dots_style_tabs'
		);

		$this->start_controls_tab(
			'dots_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-dots > .owl-dot span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'dots_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-dots > .owl-dot span:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-dots > .owl-dot span:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'dats_style_active_tab',
			[
				'label' => esc_html__('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color_active',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-dots > .owl-dot.active span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_active',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-dots > .owl-dot.active span' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'dots_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 5,
						'max'  => 50,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-dots > .owl-dot span' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'dots_border',
				'selector' => '{{WRAPPER}} .mwt-gallery-carousel-container .owl-dots > .owl-dot span',
			]
		);

		$this->add_control(
			'dots_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-dots > .owl-dot span'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'dots_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-gallery-carousel-container .owl-dots > .owl-dot span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Filter Cats
	 *
	 * Formats Category to be inserted in class attribute.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @param string $string category slug.
	 *
	 * @return string $cat_filtered slug filtered.
	 */
	public function filter_cats($string)
	{

		$cat_filtered = trim($string);

		$cat_filtered = mb_strtolower($cat_filtered);

		if (strpos($cat_filtered, 'class') || strpos($cat_filtered, 'src')) {
			$cat_filtered = substr($cat_filtered, strpos($cat_filtered, '"') + 1);
			$cat_filtered = strtok($cat_filtered, '"');
			$cat_filtered = preg_replace('/[http:.]/', '', $cat_filtered);
			$cat_filtered = str_replace('/', '', $cat_filtered);
		}

		$cat_filtered = str_replace(', ', ',', $cat_filtered);
		$cat_filtered = preg_replace('/[\s_&@!#%]/', '-', $cat_filtered);
		$cat_filtered = str_replace(',', ' ', $cat_filtered);

		return $cat_filtered;
	}

	/**
	 * Get link url.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function get_link_url($item)
	{
		return [
			'url' => wp_get_attachment_url($item['item_image']['id']),
		];
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$unique_id = uniqid();

		if (empty($settings['gallery-list'])) {
			return;
		} ?>

		<div class="mwt-gallery-carousel-container">
			<?php
			$filter = $settings['gallery_filter'];
			if ('yes' === $filter) :
			?>
				<div class="filters carousel_filters-<?php echo esc_attr($unique_id); ?>">
					<a href="#" class="category selected active" data-filter="*">
						<?php echo wp_kses_post($settings['gallery_first_cat_label']); ?>
					</a>
					<?php
					foreach ($settings['gallery_cats_content'] as $index => $category) {
						if (!empty($category['gallery_cat'])) {
							$cat_filtered = $this->filter_cats($category['gallery_cat']); ?>
							<a href="#" class="category" data-filter=".<?php echo esc_attr($cat_filtered) ?>">
								<?php echo wp_kses_post($category['gallery_cat']); ?>
							</a>
					<?php
						}
					}
					?>
				</div><!-- .filters -->
			<?php
			endif;

			$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
			$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
			$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
			$data_nav = ($settings['data_nav_prev_next'] === 'yes') ? 'true' : 'false';
			$data_dots = ($settings['data_nav_dots'] === 'yes') ? 'true' : 'false';
			$data_autoplay_hover_pause = ($settings['data_autoplay_hover_pause'] === 'yes') ? 'true' : 'false';
			$data_rtl = ($settings['data_rtl'] === 'yes') ? 'true' : 'false';

			$item_layout = $settings['items_layout'];
			?>
			<div class="owl-carousel mwt-gallery-carousel layout-<?php echo esc_attr($item_layout); ?>" data-number="<?php echo esc_attr($unique_id); ?>" data-rtl="<?php echo esc_attr($data_rtl) ?>" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-md="<?php echo esc_attr($settings['data_responsive_md']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>" data-filters=".carousel_filters-<?php echo esc_attr($unique_id); ?>">
				<?php
				switch ($item_layout) {
					case 'item-regular':
						foreach ($settings['gallery-list'] as $index => $item) { ?>
							<div class="owl-carousel-item <?php echo esc_attr($this->filter_cats($item['item_category'])) ?>">
								<div class="mwt-gallery-carousel-image-wrap">
									<?php
									$title_link = $settings['gallery-list'][$index]['item_title_link']['url'];
									$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['item_image']['id'], 'thumbnail', $settings);
									if (!$image_url && isset($item['item_image']['url'])) {
										$image_url = $item['item_image']['url'];
									}
									if (!empty($item['item_title'])) {
										$image_alt = $item['item_title'];
									} else {
										$image_alt = 'This is a title';
									}

									$link = $this->get_link_url($item);
									if ($link) {
										$link_key = 'link_' . $index;
										$this->add_lightbox_data_attributes($link_key, $item['item_image']['id'], $settings['open_lightbox'], $this->get_id());
										$this->add_link_attributes($link_key, $link);
									}
									$gallery_lightbox_icon = isset($settings['gallery_lightbox_icon']);
									$gallery_links_icon = isset($settings['gallery_links_icon']);

									if ('yes' === $settings['open_lightbox'] && !empty($gallery_lightbox_icon) || 'yes' === $settings['gallery-list'][$index]['show_link'] && !empty($gallery_links_icon)) { ?>
										<div class="links-wrap">
											<div class="links-inner-wrap">
												<?php if ('yes' === $settings['open_lightbox'] && !empty($gallery_lightbox_icon)) { ?>
													<a <?php echo wp_kses_post($this->get_render_attribute_string($link_key)) ?>>
														<?php \Elementor\Icons_Manager::render_icon($settings['gallery_lightbox_icon'], ['aria-hidden' => 'true']) ?>
													</a>
												<?php }
												if ('yes' === $settings['gallery-list'][$index]['show_link'] && !empty($gallery_links_icon)) { ?>
													<a href="<?php echo esc_url($settings['gallery-list'][$index]['item_title_link']['url']) ?>">
														<?php \Elementor\Icons_Manager::render_icon($settings['gallery_links_icon'], ['aria-hidden' => 'true']) ?>
													</a>
												<?php } ?>
											</div>
										</div><!-- .links-wrap -->
									<?php
									}
									$image_html = '<img src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
									echo wp_kses_post($image_html); ?>
								</div>
								<?php
								$category = $settings['gallery-list'][$index]['item_category'];
								$category_array = explode(", ", $category);

								$title = $settings['gallery-list'][$index]['item_title'];
								$title_link = $settings['gallery-list'][$index]['item_title_link']['url'];
								$description = $settings['gallery-list'][$index]['item_description'];
								?>
								<div class="mwt-gallery-carousel-content-wrap">
									<?php
									if (!empty($category_array)) { ?>
										<div class="mwt-gallery-carousel-categories-wrap">
											<?php
											$i = 0;
											while ($i < count($settings['gallery_cats_content'])) :
												foreach ($category_array as $index => $category_aray_item) {
													if ($category_aray_item === $settings['gallery_cats_content'][$i]['gallery_cat']) {
														$cat_link = implode($settings['gallery_cats_content'][$i]['gallery_cat_link']);
														if (!empty($cat_link)) { ?>
															<a href="<?php echo esc_attr($cat_link) ?>" class="mwt-gallery-carousel-categories">
																<?php echo esc_html($category_aray_item) ?>
															</a>
														<?php } else { ?>
															<span class="mwt-gallery-carousel-categories"><?php echo esc_html($category_aray_item) ?></span>
											<?php
														}
													}
												}
												$i++;
											endwhile; ?>
										</div>
									<?php }
									if (!empty($title)) { ?>
										<h3 class="mwt-gallery-carousel-title">
											<?php
											if (!empty($title_link)) { ?>
												<a href="<?php echo esc_attr($title_link) ?>">
													<?php echo esc_html($title); ?>
												</a>
											<?php } else {
												echo esc_html($title);
											} ?>
										</h3>
									<?php }
									if (!empty($description)) { ?>
										<div class="mwt-gallery-carousel-description">
											<?php echo esc_html($description); ?>
										</div>
									<?php } ?>
								</div>
							</div>
						<?php } ?>
						<?php break;

					case 'item-extended':
						foreach ($settings['gallery-list'] as $index => $item) { ?>
							<div class="owl-carousel-item <?php echo esc_attr($this->filter_cats($item['item_category'])) ?>">
								<?php
								$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['item_image']['id'], 'thumbnail', $settings);
								if (!$image_url && isset($item['item_image']['url'])) {
									$image_url = $item['item_image']['url'];
								}
								if (!empty($item['item_title'])) {
									$image_alt = $item['item_title'];
								} else {
									$image_alt = 'This is a title';
								}
								$image_html = '<img src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';
								echo wp_kses_post($image_html);

								$category = $settings['gallery-list'][$index]['item_category'];
								$category_array = explode(", ", $category);

								$title = $settings['gallery-list'][$index]['item_title'];
								$title_link = $settings['gallery-list'][$index]['item_title_link']['url'];
								?>
								<div class="mwt-gallery-carousel-content-wrap">
									<?php
									if (!empty($category_array)) { ?>
										<div class="mwt-gallery-carousel-categories-wrap">
											<?php
											$i = 0;
											while ($i < count($settings['gallery_cats_content'])) :
												foreach ($category_array as $index => $category_aray_item) {
													if ($category_aray_item === $settings['gallery_cats_content'][$i]['gallery_cat']) {
														$cat_link = implode($settings['gallery_cats_content'][$i]['gallery_cat_link']);
														if (!empty($cat_link)) { ?>
															<a href="<?php echo esc_attr($cat_link) ?>" class="mwt-gallery-carousel-categories">
																<?php echo esc_html($category_aray_item) ?>
															</a>
														<?php } else { ?>
															<span class="mwt-gallery-carousel-categories"><?php echo esc_html($category_aray_item) ?></span>
											<?php
														}
													}
												}
												$i++;
											endwhile; ?>
										</div>
									<?php }
									if (!empty($title)) { ?>
										<h3 class="mwt-gallery-carousel-title">
											<?php
											if (!empty($title_link)) { ?>
												<a href="<?php echo esc_attr($title_link) ?>">
													<?php echo esc_html($title); ?>
												</a>
											<?php } else {
												echo esc_html($title);
											} ?>
										</h3>
									<?php } ?>
								</div>
							</div>
						<?php } ?>
						<?php break;

					case 'item-image':
						foreach ($settings['gallery-list'] as $index => $item) { ?>
							<div class="owl-carousel-item <?php echo esc_attr($this->filter_cats($item['item_category'])) ?>">
								<div class="mwt-gallery-carousel-image-wrap">
									<?php
									$title_link = $settings['gallery-list'][$index]['item_title_link']['url'];
									$image_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($item['item_image']['id'], 'thumbnail', $settings);
									if (!$image_url && isset($item['item_image']['url'])) {
										$image_url = $item['item_image']['url'];
									}
									if (!empty($item['item_title'])) {
										$image_alt = $item['item_title'];
									} else {
										$image_alt = 'This is a title';
									}

									$link = $this->get_link_url($item);
									if ($link) {
										$link_key = 'link_' . $index;
										$this->add_lightbox_data_attributes($link_key, $item['item_image']['id'], $settings['open_lightbox'], $this->get_id());
										$this->add_link_attributes($link_key, $link);
									}
									$gallery_lightbox_icon = isset($settings['gallery_lightbox_icon']);
									$gallery_links_icon = isset($settings['gallery_links_icon']);

									if ('yes' === $settings['open_lightbox'] && !empty($gallery_lightbox_icon) || 'yes' === $settings['gallery-list'][$index]['show_link'] && !empty($gallery_links_icon)) { ?>
										<div class="links-wrap">
											<div class="links-inner-wrap">
												<?php if ('yes' === $settings['open_lightbox'] && !empty($gallery_lightbox_icon)) { ?>
													<a <?php echo wp_kses_post($this->get_render_attribute_string($link_key)) ?> class="mwt-gallery-carousel-lightbox-icon">
														<?php \Elementor\Icons_Manager::render_icon($settings['gallery_lightbox_icon'], ['aria-hidden' => 'true']) ?>
													</a>
												<?php }
												if ('yes' === $settings['gallery-list'][$index]['show_link'] && !empty($gallery_links_icon)) { ?>
													<a href="<?php echo esc_url($title_link) ?>">
														<?php \Elementor\Icons_Manager::render_icon($settings['gallery_links_icon'], ['aria-hidden' => 'true']) ?>
													</a>
												<?php } ?>
											</div>
										</div><!-- .links-wrap -->
									<?php
									}
									$image_html = '<img src="' . esc_attr($image_url) . '" alt="' . esc_attr($image_alt) . '" />';

									if (!empty($title_link) && 'yes' !== $settings['open_lightbox'] && 'yes' !== $settings['gallery-list'][$index]['show_link']) { ?>
										<a href="<?php echo esc_attr($title_link) ?>" class="mwt-gallery-carousel-title-link">
										<?php }
									echo wp_kses_post($image_html);
									if (!empty($title_link) && 'yes' !== $settings['open_lightbox'] && 'yes' !== $settings['gallery-list'][$index]['show_link']) { ?>
										</a>
									<?php } ?>
								</div>
							</div>
				<?php
						}
						break;
				} ?>
			</div><!-- .owl-carousel -->
			<div class="owl-nav position-<?php echo esc_attr($settings['prev_next_icons_position']) ?> mwt-gallery-carousel-nav-<?php echo esc_attr($unique_id); ?>">
				<?php
				$date_nav_prev = isset($settings['date_nav_prev']) ? $settings['date_nav_prev'] : '';
				$date_nav_next = isset($settings['date_nav_next']) ? $settings['date_nav_next'] : '';
				$date_nav_prev_word = isset($settings['date_nav_prev_word']) ? $settings['date_nav_prev_word'] : '';
				$date_nav_next_word = isset($settings['date_nav_next_word']) ? $settings['date_nav_next_word'] : '';
				?>
				<span class="mwt-gallery-carousel-prev-<?php echo esc_attr($unique_id); ?>">
					<?php if ($date_nav_prev) :
						\Elementor\Icons_Manager::render_icon($settings['date_nav_prev'], ['aria-hidden' => 'true', 'class' => 'icon-prev']);
					endif;
					echo esc_html($date_nav_prev_word)
					?>
				</span>
				<span class="mwt-gallery-carousel-next-<?php echo esc_attr($unique_id); ?>">
					<?php
					echo esc_html($date_nav_next_word);
					if ($date_nav_next) :
						\Elementor\Icons_Manager::render_icon($settings['date_nav_next'], ['aria-hidden' => 'true', 'class' => 'icon-next']);
					endif;
					?>
				</span>
			</div>
		</div>
<?php }
}
