<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Parallax Box Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Parallax_Box extends \Elementor\Widget_Base

{
    /**
     * Retrieve Widget Dependent CSS.
     *
     * @since 1.0.0
     * @access public
     * @return array CSS style handles.
     */
    public function get_style_depends()
    {
        return ['mwt-parallax-box-style'];
    }

    /**
     * Retrieve Widget Dependent JS.
     *
     * @since 1.0.0
     * @access public
     * @return array JS script handles.
     */
    public function get_script_depends()
    {
        return ['parallax-script'];
        return ['mwt-parallax-box-script'];
    }

    /**
     * Get widget name.
     *
     * Retrieve list widget name.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'mwt-parallax-box';
    }

    /**
     * Get widget title.
     *
     * Retrieve list widget title.
     *
     * @since 1.0.0
     * @access public
     * @return string Widget title.
     */
    public function get_title()
    {
        return esc_html__('MWT Parallax Box', 'mwt-addons-for-elementor');
    }

    /**
     * Get widget categories.
     *
     * Retrieve the list of categories the list widget belongs to.
     *
     * @since 1.0.0
     * @access public
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['mwt'];
    }

    /**
     * Get widget icon.
     *
     * Retrieve image carousel widget icon.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'eicon-parallax';
    }

    /**
     * Get widget keywords.
     *
     * Retrieve the list of keywords the widget belongs to.
     *
     * @since 2.1.0
     * @access public
     *
     * @return array Widget keywords.
     */
    public function get_keywords()
    {
        return ['mwt', 'image', 'photo', 'visual', 'before', 'after', 'parallax'];
    }

    /**
     * Register list widget controls.
     *
     * Add input fields to allow the user to customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls()
    {

        $this->start_controls_section(
            'content_section',
            [
                'label' => esc_html__('Box', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->start_controls_tabs(
            'content_tabs'
        );

        $this->start_controls_tab(
            'content_front_tab',
            [
                'label' => esc_html__('Front', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'image_front',
            [
                'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
                'ai' => [
                    'active' => false,
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'front_thumbnail',
                'separator' => 'none',
                'default' => 'full',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'content_parallax_tab',
            [
                'label' => esc_html__('Parallax Items', 'mwt-addons-for-elementor'),
            ]
        );

        /* Start repeater */

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'image_parallax',
            [
                'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
                'ai' => [
                    'active' => false,
                ],
            ]
        );

        $repeater->add_control(
            'parallax_depth',
            [
                'label' => esc_html__('Depth', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::NUMBER,
                'description' => esc_html__('Enter layer motion', 'mwt-addons-for-elementor'),
                'min' => 0.1,
                'max' => 1,
                'step' => 0.1,
                'default' => 1,
            ]
        );

        $repeater->add_responsive_control(
            'parallax_item_opacity',
            [
                'label' => esc_html__('Opacity', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'max' => 1,
                        'min' => 0.10,
                        'step' => 0.01,
                    ],
                ],
                'default' => [
                    'size' => 1,
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}}' => 'opacity: {{SIZE}};',
                ],
            ]
        );

        $repeater->add_control(
            'parallax_item_width',
            [
                'label' => esc_html__('Width', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Value', 'mwt-addons-for-elementor'),
                'label_off' => esc_html__('Auto', 'mwt-addons-for-elementor'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $repeater->add_responsive_control(
            'parallax_item_width_value',
            [
                'label' => esc_html__('Width Value', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => ['%', 'px'],
                'range' => [
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => '%',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}}' => 'width: {{SIZE}}{{UNIT}}; max-width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'parallax_item_width' => 'yes',
                ]
            ]
        );

        $repeater->add_control(
            'parallax_item_position',
            [
                'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::SELECT,
                'default' => 'scene-top-left',
                'options' => [
                    'scene-top-left' => esc_html__('Top Left', 'mwt-addons-for-elementor'),
                    'scene-top-right' => esc_html__('Top Right', 'mwt-addons-for-elementor'),
                    'scene-bottom-left'  => esc_html__('Bottom Left', 'mwt-addons-for-elementor'),
                    'scene-bottom-right' => esc_html__('Bottom Right', 'mwt-addons-for-elementor'),
                    'scene-custom' => esc_html__('Custom', 'mwt-addons-for-elementor'),
                ],
            ]
        );

        $repeater->add_responsive_control(
            'parallax_item_position_custom',
            [
                'label'      => __('Position Value', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} {{CURRENT_ITEM}}' => 'top: {{TOP}}{{UNIT}} !important; right: {{RIGHT}}{{UNIT}} !important; bottom: {{BOTTOM}}{{UNIT}} !important; left: {{LEFT}}{{UNIT}} !important;',
                ],
                'condition' => [
                    'parallax_item_position' => 'scene-custom',
                ]
            ]
        );

        /* End repeater */

        $this->add_control(
            'parallax_items_list',
            [
                'label' => esc_html__('Parallax Items', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'image_parallax' => '',
                        'parallax_depth' => '',
                        'parallax_item_position' => '',
                    ],
                ],
                'title_field' => '',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'content_back_tab',
            [
                'label' => esc_html__('Back', 'mwt-addons-for-elementor'),
            ]
        );

        $this->add_control(
            'image_back',
            [
                'label' => esc_html__('Choose Image', 'mwt-addons-for-elementor'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
                'ai' => [
                    'active' => false,
                ],
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'back_thumbnail',
                'separator' => 'none',
                'default' => 'full',
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        // style
        $this->start_controls_section(
            'style_section',
            [
                'label' => esc_html__('Box', 'mwt-addons-for-elementor'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'box_align',
            [
                'label'     => __('Alignment', 'mwt-addons-for-elementor'),
                'type'      => \Elementor\Controls_Manager::CHOOSE,
                'options'   => [
                    'left' => [
                        'title' => __('Left', 'mwt-addons-for-elementor'),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center'     => [
                        'title' => __('Center', 'mwt-addons-for-elementor'),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'   => [
                        'title' => __('Right', 'mwt-addons-for-elementor'),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
                'default'   => 'center',
                'selectors' => [
                    '{{WRAPPER}} .mwt-parallax-box-scene' => 'text-align: {{VALUE}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'box_margin',
            [
                'label'      => __('Margin', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-parallax-box-scene .mwt-parallax-box-images-wrap' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'box_back_image_position',
            [
                'label'      => __('Back Image Position', 'mwt-addons-for-elementor'),
                'type'       => \Elementor\Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .mwt-parallax-box-scene .mwt-parallax-box-images-wrap img.mwt-parallax-box-image-back' => 'top: {{TOP}}{{UNIT}}; right: {{RIGHT}}{{UNIT}}; bottom: {{BOTTOM}}{{UNIT}}; left: {{LEFT}}{{UNIT}};',
                ],
            ]
        );




        $this->end_controls_section();
    }

    /**
     * Render list widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
?>
        <div class="mwt-parallax-box-container">
            <div class="mwt-parallax-box-scene">
                <?php if (!empty($settings['parallax_items_list'])) : ?>
                    <div class="scene mwt-parallax-box-layers">
                        <?php foreach ($settings['parallax_items_list'] as $item) :
                            $depth = $item['parallax_depth'];
                            $item_positon = $item['parallax_item_position'];
                            if (!empty($item['image_parallax']['url'])) : ?>
                                <img class="elementor-repeater-item-<?php echo esc_attr($item['_id']); ?> mwt-parallax-box-layer <?php echo esc_html($item_positon) ?>" src="<?php echo esc_url($item['image_parallax']['url']) ?>" alt="<?php echo \Elementor\Control_Media::get_image_alt($item['image_parallax']) ?>" data-depth="<?php echo esc_attr($depth) ?>">
                        <?php endif;
                        endforeach; ?>
                    </div>
                <?php endif; ?>
                <div class="mwt-parallax-box-images-wrap">
                    <?php
                    if (!empty($settings['image_front']['url'])) :
                        $image_front_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($settings['image_front']['id'], 'front_thumbnail', $settings);
                        if (!$image_front_url && isset($settings['image_front']['url'])) {
                            $image_front_url = $settings['image_front']['url'];
                        }
                        $image_front_html = '<img class="mwt-parallax-box-image-front" src="' . esc_attr($image_front_url) . '" alt="' . \Elementor\Control_Media::get_image_alt($settings['image_front']) . '" />';
                        echo wp_kses_post($image_front_html);
                    endif;
                    if (!empty($settings['image_back']['url'])) :
                        $image_back_url = \Elementor\Group_Control_Image_Size::get_attachment_image_src($settings['image_back']['id'], 'back_thumbnail', $settings);
                        if (!$image_back_url && isset($settings['image_back']['url'])) {
                            $image_back_url = $settings['image_back']['url'];
                        }
                        $image_back_html = '<img class="mwt-parallax-box-image-back" src="' . esc_attr($image_back_url) . '" alt="' . \Elementor\Control_Media::get_image_alt($settings['image_back']) . '" />';
                        echo wp_kses_post($image_back_html);
                    endif; ?>
                </div>
            </div>
        </div>
<?php }
}
