<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor MWT Posts Widget.
 *
 * Elementor widget that inserts an embbedable content into the page, from any given URL.
 *
 * @since 1.0.0
 */

class Elementor_MWT_Posts extends \Elementor\Widget_Base

{

	/**
	 * Retrieve Widget Dependent JS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array JS script handles.
	 */
	public function get_script_depends()
	{
		return ['mwt-posts-script'];
	}

	/**
	 * Retrieve Widget Dependent CSS.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array CSS style handles.
	 */
	public function get_style_depends()
	{
		return ['owl-carousel-style'];
		return ['mwt-posts-style'];
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'mwt-posts';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return esc_html__('MWT Posts', 'mwt-addons-for-elementor');
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['mwt'];
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve image post widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'eicon-single-post';
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the widget belongs to.
	 *
	 * @since 2.1.0
	 * @access public
	 *
	 * @return array Widget keywords.
	 */
	public function get_keywords()
	{
		return ['mwt', 'visual', 'posts', 'carousel', 'grid', 'slider'];
	}

	/**
	 * Register list widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls()
	{

		$this->start_controls_section(
			'general_settings',
			[
				'label' => __('General', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'posts_layout',
			[
				'label' => esc_html__('Posts Layout', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'carousel',
				'options' => [
					'carousel' => esc_html__('Carousel', 'mwt-addons-for-elementor'),
					'grid' => esc_html__('Masonry Grid', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'item_layout',
			[
				'label' => esc_html__('Item Layout', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => '',
				'options' => [
					'' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'side-item' => esc_html__('Side', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
				'condition' => [
					'posts_layout' => 'grid',
				]
			]
		);

		$this->add_responsive_control(
			'left_side_item_width',
			[
				'label' => esc_html__('Left side width', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'description' => __('\'Left side width\' is active when screen size > 991px', 'mwt-addons-for-elementor'),
				'size_units' => ['%', 'px'],
				'range' => [
					'%' => [
						'min' => 30,
						'max' => 70,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 50,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-grid .grid-item.side-item .mwt-posts-article .mwt-posts-media-content' => 'flex-basis: {{SIZE}}{{UNIT}}; width: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-posts-grid .grid-item.side-item .mwt-posts-article .mwt-posts-item-content' => 'flex-basis: calc(100% - {{SIZE}}{{UNIT}}); width: calc(100% - {{SIZE}}{{UNIT}});',
				],
				'condition' => [
					'item_layout' => 'side-item',
				]
			]
		);

		$this->add_control(
			'number_of_posts',
			[
				'label'       => __('Posts Per Page', 'mwt-addons-for-elementor'),
				'description' => __('Set the number of per page', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::NUMBER,
				'min'         => 1,
				'default'     => 3,
			]
		);

		$this->add_control(
			'posts_offset',
			[
				'label'       => __('Offset', 'mwt-addons-for-elementor'),
				'description' => __('Set the number of per page', 'mwt-addons-for-elementor'),
				'type'        => \Elementor\Controls_Manager::NUMBER,
				'min'         => 0,
				'default'     => 0,
			]
		);

		function mwt_post_type_categories()
		{
			$terms = '';
			$terms = get_terms(
				array(
					'taxonomy' => 'category',
					'hide_empty' => true,
				)
			);
			$categories = array();
			if (!empty($terms) && !is_wp_error($terms)) {
				foreach ($terms as $term) {
					$categories[$term->term_id] = $term->slug;
				}
			}
			return $categories;
		}

		$this->add_control(
			'posts_filter_cat',
			[
				'label'         => __('Filter By Category', 'mwt-addons-for-elementor'),
				'type'          => \Elementor\Controls_Manager::SELECT2,
				'label_block'   => true,
				'multiple'      => true,
				'options'       => mwt_post_type_categories(),
			]
		);

		$this->add_control(
			'posts_filter',
			[
				'label'   => __('Filter Tabs', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'posts_first_cat_label',
			[
				'label'     => __('First Category Label', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'default'   => __('All', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'posts_filter' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'posts_filters_align',
			[
				'label'     => __('Category Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .filters' => 'text-align: {{VALUE}}',
				],
				'condition'   => [
					'posts_filter' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__('Items', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'show_featured_image',
			[
				'label'     => __('Show Featured Image', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Image_Size::get_type(),
			[
				'name' => 'featured_image',
				'exclude' => ['custom'],
				'include' => [],
				'default' => 'full',
				'condition' => [
					'show_featured_image' => 'yes',
				]
			]
		);

		$this->add_control(
			'meta_data_options',
			[
				'label' => esc_html__('Meta Data', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'meta_separator',
			[
				'label' => esc_html__('Meta separator', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__('|', 'mwt-addons-for-elementor'),
				'placeholder' => esc_html__('Enter your separator here', 'mwt-addons-for-elementor'),
				'ai' => [
					'active' => false,
				],
			]
		);

		$this->add_control(
			'show_categories',
			[
				'label'     => __('Show categories', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'categories_icon',
			[
				'label'            => __('Categories icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-paperclip',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'paperclip',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'show_categories' => 'yes',
				]
			]
		);

		$this->add_control(
			'text_before_categories',
			[
				'label' => esc_html__('Text before categories', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => '',
				'placeholder' => esc_html__('Enter your text here', 'mwt-addons-for-elementor'),
				'ai' => [
					'active' => false,
				],
				'condition' => [
					'show_categories' => 'yes',
				]
			]
		);

		$this->add_control(
			'categories_separator',
			[
				'label' => esc_html__('Categories separator', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__(',', 'mwt-addons-for-elementor'),
				'placeholder' => esc_html__('Enter your separator here', 'mwt-addons-for-elementor'),
				'ai' => [
					'active' => false,
				],
				'condition' => [
					'show_categories' => 'yes',
				]
			]
		);

		$this->add_control(
			'show_categories_over_thumbnail',
			[
				'label'     => __('Show Categories Over Thumbnail', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_featured_image' => 'yes',
					'show_categories' => 'yes',
				],
			]
		);

		$this->add_control(
			'show_date',
			[
				'label'     => __('Show date', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'date_icon',
			[
				'label'            => __('Date icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-calendar-alt',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'calendar-alt',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'show_date' => 'yes',
				]
			]
		);

		$this->add_control(
			'text_before_date',
			[
				'label' => esc_html__('Text before date', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => '',
				'placeholder' => esc_html__('Enter your text here', 'mwt-addons-for-elementor'),
				'ai' => [
					'active' => false,
				],
				'condition' => [
					'show_date' => 'yes',
				]
			]
		);

		$this->add_control(
			'show_author',
			[
				'label'     => __('Show author', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'author_icon',
			[
				'label'            => __('Author icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-user',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'user',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'show_author' => 'yes',
				]
			]
		);

		$this->add_control(
			'text_before_author',
			[
				'label' => esc_html__('Text before author', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => '',
				'placeholder' => esc_html__('Enter your text here', 'mwt-addons-for-elementor'),
				'ai' => [
					'active' => false,
				],
				'condition' => [
					'show_author' => 'yes',
				]
			]
		);

		$this->add_control(
			'show_author_avatar',
			[
				'label'     => __('Show author avatar', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_author' => 'yes',
				]
			]
		);

		$this->add_control(
			'show_comments_count',
			[
				'label'     => __('Show comments count', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'comments_count_icon',
			[
				'label'            => __('Comments count icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-comments',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'comments',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'show_comments_count' => 'yes',
				]
			]
		);

		$this->add_control(
			'text_before_comments_count',
			[
				'label' => esc_html__('Text before comments count', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => '',
				'placeholder' => esc_html__('Enter your text here', 'mwt-addons-for-elementor'),
				'ai' => [
					'active' => false,
				],
				'condition' => [
					'show_comments_count' => 'yes',
				]
			]
		);

		$this->add_control(
			'show_excerpt',
			[
				'label'     => __('Show excerpt', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'posts_excerpt_length',
			[
				'label'     => __('Excerpt length', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::NUMBER,
				'default'   => 55,
				'condition' => [
					'show_excerpt'   => 'yes',
				]
			]
		);

		$this->add_control(
			'show_more_link',
			[
				'label'     => __('Read More', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
			]
		);

		$this->add_control(
			'more_link_icon',
			[
				'label'            => __('Read More icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-angle-right',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'angle-right',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'show_more_link' => 'yes',
				]
			]
		);

		$this->add_control(
			'footer_block_options',
			[
				'label' => esc_html__('Footer Block', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'show_footer',
			[
				'label'     => __('Show footer block', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'no',
			]
		);

		$this->add_control(
			'footer_meta_separator',
			[
				'label' => esc_html__('Meta separator', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__('|', 'mwt-addons-for-elementor'),
				'placeholder' => esc_html__('Enter your separator here', 'mwt-addons-for-elementor'),
				'ai' => [
					'active' => false,
				],
				'condition' => [
					'show_footer' => 'yes',
				]
			]
		);

		$this->add_control(
			'footer_show_categories',
			[
				'label'     => __('Show categories', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'separator' => 'before',
				'condition' => [
					'show_footer' => 'yes',
					'show_categories' => 'yes',
				]
			]
		);

		$this->add_control(
			'footer_show_date',
			[
				'label'     => __('Show date', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_footer' => 'yes',
					'show_date' => 'yes',
				]
			]
		);

		$this->add_control(
			'footer_show_author',
			[
				'label'     => __('Show author', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_footer' => 'yes',
					'show_author' => 'yes',
				]
			]
		);

		$this->add_control(
			'footer_show_comments_count',
			[
				'label'     => __('Show comments count', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => [
					'show_footer' => 'yes',
					'show_comments_count' => 'yes',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'carousel_settings',
			[
				'label' => __('Carousel Settings', 'mwt-addons-for-elementor'),
				'condition' => [
					'posts_layout' => 'carousel',
				]
			]
		);

		$this->add_control(
			'data_loop',
			[
				'label'   => __('Loop', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_rtl',
			[
				'label'   => __('Enable RTL', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay',
			[
				'label'   => __('Autoplay', 'mwt-addons-for-elementor'),
				'description' => esc_html__('Slide will start automatically', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplaytimeout',
			[
				'label' => esc_html__('Autoplay Speed (ms)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 5000,
				'min' => 0,
				'step' => 50,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_autoplay_hover_pause',
			[
				'label'   => __('Pause on Hover', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_center',
			[
				'label'   => __('Centered Carousel?', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
			]
		);

		//Slides to Scroll
		$this->add_control(
			'data_responsive_lg',
			[
				'label' => esc_html__('Desktop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 4,
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_md',
			[
				'label' => esc_html__('Laptop Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_sm',
			[
				'label' => esc_html__('Tablet Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 2,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_responsive_xs',
			[
				'label' => esc_html__('Mobile Slides', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 1,
				'max' => 12,
				'step' => 1,
				'default' => 1,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_margin',
			[
				'label' => esc_html__('Item Gap (px)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 15,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_margin',
			[
				'label' => esc_html__('Item Gap (px)', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'min' => 0,
				'max' => 80,
				'step' => 1,
				'default' => 30,
				'separator' => 'before',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'data_nav_prev_next',
			[
				'label'   => __('Show prev/next', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'date_nav_prev',
			[
				'label'            => __('Previous Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-chevron-left',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'chevron-left',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'date_nav_prev_word',
			[
				'label'     => __('Previous word', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Prev', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'data_nav_prev_next' => 'yes',
				],
			]
		);

		$this->add_control(
			'date_nav_next',
			[
				'label'            => __('Next Icon', 'mwt-addons-for-elementor'),
				'type'             => \Elementor\Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
				'default'          => [
					'value'   => 'fas fa-chevron-right',
					'library' => 'fa-solid',
				],
				'recommended'      => [
					'fa-solid'   => [
						'chevron-right',
					],
				],
				'skin'             => 'inline',
				'label_block'      => false,
				'condition' => [
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'date_nav_next_word',
			[
				'label'     => __('Next word', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__('Next', 'mwt-addons-for-elementor'),
				'dynamic' => [
					'active' => true,
				],
				'condition'   => [
					'data_nav_prev_next' => 'yes',
				],
			]
		);

		$this->add_control(
			'data_nav_dots',
			[
				'label'   => __('Show dots', 'mwt-addons-for-elementor'),
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'default' => 'yes',
				'frontend_available' => true,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'posts_grid_settings',
			[
				'label' => __('Grid Settings', 'mwt-addons-for-elementor'),
				'condition' => [
					'posts_layout'   => 'grid',
				]
			]
		);

		$this->add_control(
			'posts_items_per_row',
			[
				'label' => esc_html__('Items per row', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'description' => esc_html__('How many items should be shown per row?', 'mwt-addons-for-elementor'),
				'min'  => 1,
				'max'  => 4,
				'step' => 1,
				'default' => 3,
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'posts_layout_gap',
			[
				'label' => esc_html__('Gap', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'solid',
				'options' => [
					'30' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'0' => esc_html__('None', 'mwt-addons-for-elementor'),
					'1'  => esc_html__('1px', 'mwt-addons-for-elementor'),
					'2' => esc_html__('2px', 'mwt-addons-for-elementor'),
					'3' => esc_html__('3px', 'mwt-addons-for-elementor'),
					'4' => esc_html__('4px', 'mwt-addons-for-elementor'),
					'5' => esc_html__('5px', 'mwt-addons-for-elementor'),
					'10' => esc_html__('10px', 'mwt-addons-for-elementor'),
					'15' => esc_html__('15px', 'mwt-addons-for-elementor'),
					'20' => esc_html__('20px', 'mwt-addons-for-elementor'),
					'30' => esc_html__('30px', 'mwt-addons-for-elementor'),
					'40' => esc_html__('40px', 'mwt-addons-for-elementor'),
					'45' => esc_html__('45px', 'mwt-addons-for-elementor'),
					'50' => esc_html__('50px', 'mwt-addons-for-elementor'),
					'60' => esc_html__('60px', 'mwt-addons-for-elementor'),
				],
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'posts_horizontal_order',
			[
				'label' => esc_html__('Horizontal Order', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SWITCHER,
				'label_on' => esc_html__('Yes', 'mwt-addons-for-elementor'),
				'label_off' => esc_html__('No', 'mwt-addons-for-elementor'),
				'return_value' => 'yes',
				'default' => 'no',
			]
		);

		$this->end_controls_section();

		// style
		$this->start_controls_section(
			'style_item_section',
			[
				'label' => esc_html__('Item', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'item_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-article'  => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'item_align',
			[
				'label'     => __('Align Items', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'start' => [
						'title' => __('Start', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-start-h',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-center-h',
					],
					'end'   => [
						'title' => __('End', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-align-end-h',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-article .mwt-posts-item-content' => 'text-align: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'item_border',
				'selector' => '{{WRAPPER}} .mwt-posts-article',
			]
		);

		$this->add_control(
			'item_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-article' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'outer_item_shadow',
				'selector' => '{{WRAPPER}} .mwt-posts-article',
			]
		);

		$this->add_control(
			'item_media_options',
			[
				'label' => esc_html__('Media', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'item_media_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-article .mwt-posts-media-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'item_content_options',
			[
				'label' => esc_html__('Content', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'item_content_align',
			[
				'label'     => __('Align Content', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'start' => [
						'title' => __('Start', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-justify-start-v',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-justify-center-v',
					],
					'end'   => [
						'title' => __('End', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-justify-end-v',
					],
				],
				'default'   => 'center',
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-grid .grid-item.side-item .mwt-posts-article .mwt-posts-item-content' => 'justify-content: {{VALUE}}',
				],
				'condition' => [
					'posts_layout' => 'grid',
					'item_layout' => 'side-item',
				]
			]
		);

		$this->add_control(
			'item_content_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-article .mwt-posts-item-content'  => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'item_content_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-article .mwt-posts-item-content' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'item_content_border',
				'selector' => '{{WRAPPER}} .mwt-posts-article .mwt-posts-item-content',
			]
		);

		$this->add_responsive_control(
			'item_content_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-article .mwt-posts-item-content' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'item_content_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-article .mwt-posts-item-content' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'item_content_shadow',
				'selector' => '{{WRAPPER}} .mwt-posts-article .mwt-posts-item-content',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_image_section',
			[
				'label' => esc_html__('Image', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition'   => [
					'show_featured_image' => 'yes',
				],
			]
		);

		$this->add_control(
			'image_hover_overlay_color',
			[
				'label' => esc_html__('Overlay Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-article .mwt-posts-thumbnail:before' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'image_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-article .mwt-posts-thumbnail:before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .mwt-posts-article .mwt-posts-thumbnail img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_categories_section',
			[
				'label' => esc_html__('Categories', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_categories' => 'yes',
					'show_categories_over_thumbnail' => 'yes',
				]
			]
		);

		$this->add_control(
			'categories_position',
			[
				'label' => esc_html__('Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'top-left',
				'options' => [
					'top-left' => esc_html__('Top Left', 'mwt-addons-for-elementor'),
					'top-right' => esc_html__('Top Right', 'mwt-addons-for-elementor'),
					'bottom-left'  => esc_html__('Bottom Left', 'mwt-addons-for-elementor'),
					'bottom-right' => esc_html__('Bottom Right', 'mwt-addons-for-elementor'),
				],
			]
		);

		$this->add_control(
			'categories_position_top',
			[
				'label' => esc_html__('Top', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-media-content .mwt-posts-categories' => 'top: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'categories_position' => ['top-left', 'top-right'],
				]
			]
		);

		$this->add_control(
			'categories_position_bottom',
			[
				'label' => esc_html__('Bottom', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => -500,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => -50,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-media-content .mwt-posts-categories' => 'bottom: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'categories_position' => ['bottom-left', 'bottom-right'],
				]
			]
		);

		$this->add_control(
			'categories_position_left',
			[
				'label' => esc_html__('Left', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-media-content .mwt-posts-categories' => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'categories_position' => ['top-left', 'bottom-left'],
				]
			]
		);

		$this->add_control(
			'categories_position_right',
			[
				'label' => esc_html__('Right', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-media-content .mwt-posts-categories' => 'right: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'categories_position' => ['top-right', 'bottom-right'],
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'categories_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-posts-categories a',
			]
		);

		$this->start_controls_tabs(
			'categories_style_tabs'
		);

		$this->start_controls_tab(
			'categories_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'categories_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-categories a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'categories_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-categories a' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'categories_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'categories_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-categories a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'categories_background_color_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-categories a:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'categories_border_color_hover',
			[
				'label'     => __('Border Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-categories a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'categories_border',
				'selector' => '{{WRAPPER}} .mwt-posts-categories a',
			]
		);

		$this->add_control(
			'categories_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-categories a'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'categories_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-categories a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'categories_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-categories a' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_meta_section',
			[
				'label' => esc_html__('Meta', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'conditions' => [
					'relation' => 'or',
					'terms' => [
						[
							'name' => 'show_categories',
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => 'show_date',
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => 'show_author',
							'operator' => '===',
							'value' => 'yes',
						],
						[
							'name' => 'show_comments_count',
							'operator' => '===',
							'value' => 'yes',
						],
					],
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'meta_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item, {{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item a',
				'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item, {{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item a',
			]
		);

		$this->start_controls_tabs(
			'meta_style_tabs'
		);

		$this->start_controls_tab(
			'meta_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'meta_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item a' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_color_categories',
			[
				'label' => esc_html__('Categories color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'separator' => 'before',
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-categories' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-categories a' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-categories' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-categories a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_color_date',
			[
				'label' => esc_html__('Date color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-date' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-date a' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-date' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-date a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_color_author',
			[
				'label' => esc_html__('Author color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-author' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-author a' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-author' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-author a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_color_comments_count',
			[
				'label' => esc_html__('Comments count color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-comments-count' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-comments-count a' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-comments-count' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-comments-count a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'meta_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'meta_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item a:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_color_hover_categories',
			[
				'label' => esc_html__('Categories color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'separator' => 'before',
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-categories a:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-categories a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_color_hover_date',
			[
				'label' => esc_html__('Date color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-date a:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-date a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_color_hover_author',
			[
				'label' => esc_html__('Author color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-author a:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-author a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'meta_color_hover_comments_count',
			[
				'label' => esc_html__('Comments count color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta .mwt-posts-meta-item.mwt-posts-comments-count a:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-footer .mwt-posts-meta-item.mwt-posts-comments-count a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'meta_spacing',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-meta'  => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'meta_icon',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'meta_icon_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-meta-icon' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-meta-item svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'meta_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 64,
						'step' => 1,
					],
					'em' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 14,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-meta-icon'  => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-posts-meta-item svg'  => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'meta_icon_margin_right',
			[
				'label'      => __('Right margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-meta-icon'  => 'margin-right: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-posts-meta-item svg'  => 'margin-right: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'meta_separator_style',
			[
				'label' => esc_html__('Separator', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'meta_separator_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-separator' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'meta_separator_spacing',
			[
				'label'      => __('Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 4,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-separator'  => 'margin-left: {{SIZE}}{{UNIT}}; margin-right: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'meta_author_avatar',
			[
				'label' => esc_html__('Author Avatar', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
				'condition'   => [
					'show_author' => 'yes',
					'show_author_avatar' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'meta_author_avatar_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px'],
				'range' => [
					'px' => [
						'min' => 30,
						'max' => 100,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 60,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-author-avatar img'  => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
				'condition'   => [
					'show_author' => 'yes',
					'show_author_avatar' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'meta_author_avatar_left_position',
			[
				'label'      => __('Left position', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-author-avatar'  => 'left: {{SIZE}}{{UNIT}};',
				],
				'condition'   => [
					'show_author' => 'yes',
					'show_author_avatar' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'meta_author_avatar_bottom_position',
			[
				'label'      => __('Bottom position', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 1,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-author-avatar'  => 'bottom: {{SIZE}}{{UNIT}};',
				],
				'condition'   => [
					'show_author' => 'yes',
					'show_author_avatar' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_title_section',
			[
				'label' => esc_html__('Title', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'title_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-posts-item-content .mwt-posts-title',
				'{{WRAPPER}} .mwt-posts-item-content .mwt-posts-title > a',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-item-content .mwt-posts-title > a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'title_color_hover',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-item-content .mwt-posts-title > a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'title_height',
			[
				'label'      => __('Height', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'default'    => [
					'unit' => 'px',
					'size' => 60,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-item-content .mwt-posts-title'  => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'title_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-item-content .mwt-posts-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_excerpt_section',
			[
				'label' => esc_html__('Excerpt', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_excerpt' => 'yes',
				]
			]
		);

		$this->add_responsive_control(
			'excerpt_align',
			[
				'label'     => __('Alignment', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::CHOOSE,
				'options'   => [
					'left' => [
						'title' => __('Left', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-left',
					],
					'center'     => [
						'title' => __('Center', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-center',
					],
					'right'   => [
						'title' => __('Right', 'mwt-addons-for-elementor'),
						'icon'  => 'eicon-text-align-right',
					],
				],
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-excerpt' => 'text-align: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'excerpt_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-posts-excerpt',
			]
		);

		$this->add_control(
			'excerpt_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-excerpt' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'excerpt_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-excerpt' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_btn_section',
			[
				'label' => esc_html__('Read More', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition'   => [
					'show_more_link' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'btn_typography',
				'selector' =>
				'{{WRAPPER}} .mwt-posts-read-more-btn a',
			]
		);

		$this->start_controls_tabs(
			'btn_style_tabs'
		);

		$this->start_controls_tab(
			'btn_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'btn_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'btn_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'btn_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'btn_color_active',
			[
				'label' => esc_html__('Color Hover', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'btn_background_color_hover',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'btn_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'btn_border',
				'selector' => '{{WRAPPER}} .mwt-posts-read-more-btn a',
			]
		);

		$this->add_control(
			'btn_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'btn_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'btn_min_width',
			[
				'label'      => __('Min Width', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a'  => 'min-width: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->add_responsive_control(
			'btn_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-read-more-btn' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'btn_icon_heading',
			[
				'label' => esc_html__('Icon', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'btn_icon_color',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-read-more-btn a svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'btn_icon_hover_color',
			[
				'label' => esc_html__('Hover Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a:hover i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-read-more-btn a:hover svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'btn_icon_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 64,
						'step' => 1,
					],
					'em' => [
						'min' => 0,
						'max' => 10,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 14,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a i'   => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-posts-read-more-btn a svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'btn_icon_spacing',
			[
				'label'      => __('Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-read-more-btn a i'  => 'margin-left: {{SIZE}}{{UNIT}}',
					'{{WRAPPER}} .mwt-posts-read-more-btn a svg'  => 'margin-left: {{SIZE}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_footer_section',
			[
				'label' => esc_html__('Footer', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'show_footer' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'footer_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-footer' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'footer_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-footer' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'footer_border',
				'selector' => '{{WRAPPER}} .mwt-posts-footer',
			]
		);

		$this->add_responsive_control(
			'footer_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-footer' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_prev_next_section',
			[
				'label' => esc_html__('Prev/Next', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'posts_layout' => 'carousel',
					'data_nav_prev_next' => 'yes',
				]
			]
		);

		$this->add_control(
			'prev_next_icons_position',
			[
				'label' => esc_html__('Prev/Next Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'bottom',
				'options' => [
					'bottom' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'middle' => esc_html__('Middle', 'mwt-addons-for-elementor'),
				],
			]
		);

		$this->add_control(
			'prev_next_align',
			[
				'label' => esc_html__('Alignment', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__('Start', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-start-h',
					],
					'center' => [
						'title' => esc_html__('Center', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-center-h',
					],
					'end' => [
						'title' => esc_html__('End', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-align-end-h',
					],
					'space-between' => [
						'title' => esc_html__('Space Between', 'mwt-addons-for-elementor'),
						'icon' => 'eicon-justify-space-between-h',
					],
				],
				'default' => 'start',
				'toggle' => true,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav' => 'justify-content: {{VALUE}};',
				],
				'condition' => [
					'prev_next_icons_position' => 'bottom',
				]
			]
		);

		$this->add_responsive_control(
			'prev_next_offset',
			[
				'label'      => __('Offset', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 1000,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav' => 'margin-left: -{{SIZE}}{{UNIT}}; margin-right: -{{SIZE}}{{UNIT}}; width: calc(100% + {{SIZE}}{{UNIT}} * 2);',
				],
				'condition' => [
					'prev_next_icons_position' => 'bottom',
				]
			]
		);

		$this->add_responsive_control(
			'prev_next_middle_offset',
			[
				'label'      => __('Offset', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'range'      => [
					'px' => [
						'min'  => -1000,
						'max'  => 1000,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 0,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav.position-middle button.owl-prev' => 'left: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav.position-middle button.owl-next' => 'right: {{SIZE}}{{UNIT}};',
				],
				'condition' => [
					'prev_next_icons_position' => 'middle',
				]
			]
		);

		$this->add_control(
			'hr',
			[
				'type' => \Elementor\Controls_Manager::DIVIDER,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'label' => esc_html__('Typography', 'mwt-addons-for-elementor'),
				'name' => 'prev_next_typography',
				'selector' => '{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button',
			]
		);

		$this->add_responsive_control(
			'prev_next_icon_size',
			[
				'label'      => __('Icon Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 4,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button svg' => 'font-size: {{SIZE}}{{UNIT}};',

				],
			]
		);

		$this->add_control(
			'prev_next_icon_position',
			[
				'label' => esc_html__('Icon Position', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'row',
				'options' => [
					'row' => esc_html__('Default', 'mwt-addons-for-elementor'),
					'row-reverse' => esc_html__('Reverse', 'mwt-addons-for-elementor'),
				],
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button>span' => 'flex-direction: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_icon_spacing',
			[
				'label'      => __('Icon Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 12,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button>span' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs(
			'prev_next_style_tabs'
		);

		$this->start_controls_tab(
			'prev_next_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color',
			[
				'label' => esc_html__('Text Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'prev_next_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color_hover',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color_hover',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button:hover i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button:hover svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'prev_next_style_disabled_tab',
			[
				'label' => esc_html__('Disabled', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'prev_next_color_disabled',
			[
				'label' => esc_html__('Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button.disabled' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button.disabled svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_icon_color_disabled',
			[
				'label' => esc_html__('Icon Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button.disabled i' => 'color: {{VALUE}};',
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button.disabled svg' => 'fill: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_bg_color_disabled',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button.disabled' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'prev_next_border_color_disabled',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button.disabled' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'prev_next_border',
				'selector' => '{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button',
			]
		);

		$this->add_control(
			'prev_next_border_radius',
			[
				'label'      => esc_html__('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_padding',
			[
				'label'      => esc_html__('Pdding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_margin',
			[
				'label'      => esc_html__('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'allowed_dimensions' => 'vertical',
				'placeholder' => [
					'top' => '0',
					'right' => '0',
					'bottom' => '0',
					'left' => '0',
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav' => 'margin-top: {{TOP}}{{UNIT}}; margin-bottom: {{BOTTOM}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'prev_next_btn_gap',
			[
				'label'      => __('Buttons Spacing', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 16,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-nav' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_dots_section',
			[
				'label' => esc_html__('Dots', 'mwt-addons-for-elementor'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'posts_layout' => 'carousel',
					'data_nav_dots' => 'yes',
				]
			]
		);

		$this->start_controls_tabs(
			'dots_style_tabs'
		);

		$this->start_controls_tab(
			'dots_style_normal_tab',
			[
				'label' => esc_html__('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-dots > .owl-dot span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'dots_style_hover_tab',
			[
				'label' => esc_html__('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color_hover',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-dots > .owl-dot span:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_hover',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-dots > .owl-dot span:hover' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'dats_style_active_tab',
			[
				'label' => esc_html__('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'dots_color_active',
			[
				'label' => esc_html__('Background Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-dots > .owl-dot.active span' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'dots_border_color_active',
			[
				'label' => esc_html__('Border Color', 'mwt-addons-for-elementor'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-dots > .owl-dot.active span' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'dots_size',
			[
				'label'      => __('Size', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em'],
				'range'      => [
					'px' => [
						'min'  => 5,
						'max'  => 50,
						'step' => 1,
					],
				],
				'default'    => [
					'unit' => 'px',
					'size' => 10,
				],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-dots > .owl-dot span' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'dots_border',
				'selector' => '{{WRAPPER}} .mwt-posts-carousel-container .owl-dots > .owl-dot span',
			]
		);

		$this->add_control(
			'dots_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-dots > .owl-dot span'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'dots_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .owl-dots > .owl-dot span' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'posts_filter_style',
			[
				'label'     => __('Filter Tabs', 'mwt-addons-for-elementor'),
				'tab'       => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'posts_filter' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'posts_filter_typography',
				'selector' => '{{WRAPPER}} .filters a.mwt-posts-category-tab',
			]
		);

		$this->start_controls_tabs('posts_filters');

		$this->start_controls_tab(
			'posts_filters_normal',
			[
				'label' => __('Normal', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'posts_filter_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'posts_background_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'posts_filter_border',
				'selector' => '{{WRAPPER}} .filters a.mwt-posts-category-tab',
			]
		);

		$this->add_control(
			'posts_filter_border_radius',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_tab();

		$this->start_controls_tab(
			'posts_filters_hover',
			[
				'label' => __('Hover', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'posts_filter_hover_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'posts_background_hover_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab:hover' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'posts_filter_border_hover',
				'selector' => '{{WRAPPER}} .filters a.mwt-posts-category-tab:hover',
			]
		);

		$this->add_control(
			'posts_filter_border_radius_hover',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab:hover'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'posts_filters_active',
			[
				'label' => __('Active', 'mwt-addons-for-elementor'),
			]
		);

		$this->add_control(
			'posts_filter_active_color',
			[
				'label'     => __('Text Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab.active' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'posts_background_active_color',
			[
				'label'     => __('Background Color', 'mwt-addons-for-elementor'),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab.active' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name'     => 'posts_filter_border_active',
				'selector' => '{{WRAPPER}} .filters a.mwt-posts-category-tab.active',
			]
		);

		$this->add_control(
			'posts_filter_border_radius_active',
			[
				'label'      => __('Border Radius', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab.active'  => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'posts_filter_padding',
			[
				'label'      => __('Padding', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'posts_filter_margin',
			[
				'label'      => __('Margin', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .filters a.mwt-posts-category-tab' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'posts_filter_shadow',
				'selector' => '{{WRAPPER}} .filters a.mwt-posts-category-tab',
			]
		);

		$this->add_responsive_control(
			'posts_filters_margin',
			[
				'label'      => __('Margin bottom tabs', 'mwt-addons-for-elementor'),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', 'em', '%'],
				'selectors'  => [
					'{{WRAPPER}} .mwt-posts-carousel-container .filters' => 'margin-bottom: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .mwt-posts-grid .filters' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();

		$unique_id = uniqid();

		$posts_cat_array = (!empty($settings['posts_filter_cat'])) ? implode(",", $settings['posts_filter_cat']) : '-11';

		$mwt_posts = new WP_Query(array(
			'post_type'           => 'post',
			'cat' 				  => $posts_cat_array,
			'orderby'             => 'post_date',
			'order '              => 'DESC',
			'ignore_sticky_posts' => true,
			'posts_per_page'      => $settings['number_of_posts'],
			'offset' => $settings['posts_offset'],
			'tax_query' => array(
				array(
					'taxonomy' => 'post_format',
					'field' => 'slug',
					'terms' => array('post-format-quote', 'post-format-status', 'post-format-link', 'post-format-aside', 'post-format-chat'),
					'operator' => 'NOT IN'
				)
			)
		)); ?>
		<?php
		//carousel layout
		if ($settings['posts_layout'] === 'carousel') : ?>
			<div class="mwt-posts-carousel-container">
			<?php endif; //end carousel layout 

		//grid layout
		if ($settings['posts_layout'] === 'grid') : ?>
				<div class="mwt-posts-grid">
				<?php endif; //end grid layout 

			$filter = $settings['posts_filter'];
			if ('yes' === $filter) :
				//get unique terms only for posts that are showing
				$showing_terms = array();
				foreach ($mwt_posts->posts as $post) {
					foreach (get_the_terms($post->ID, 'category') as $post_term) {
						$showing_terms[$post_term->term_id] = $post_term;
					}
				} ?>
					<div class="filters filters-<?php echo esc_attr($unique_id); ?>">
						<a href="#" class="mwt-posts-category-tab selected active" data-filter="*">
							<?php echo wp_kses_post($settings['posts_first_cat_label']); ?>
						</a>
						<?php
						foreach ($showing_terms as $term_key => $term_id) {
							$current_term = get_term($term_id, 'category');
							//start carousel layout 
							if ($settings['posts_layout'] === 'carousel') : ?>
								<a class="mwt-posts-category-tab" href="#" data-filter=".<?php echo esc_attr($current_term->slug); ?>"><?php echo esc_html($current_term->name); ?></a>
							<?php endif;
							//end carousel layout 

							//start grid layout 
							if ($settings['posts_layout'] === 'grid') : ?>
								<a class="mwt-posts-category-tab" href="#" data-filter="<?php echo esc_attr($current_term->slug); ?>"><?php echo esc_html($current_term->name); ?></a>
							<?php endif;
							//end grid layout 
							?>
						<?php } ?>
					</div>
				<?php endif;

			//carousel layout
			if ($settings['posts_layout'] === 'carousel') :
				$data_loop = ($settings['data_loop'] === 'yes') ? 'true' : 'false';
				$data_autoplay = ($settings['data_autoplay'] === 'yes') ? 'true' : 'false';
				$data_center = ($settings['data_center'] === 'yes') ? 'true' : 'false';
				$data_nav = ($settings['data_nav_prev_next'] === 'yes') ? 'true' : 'false';
				$data_dots = ($settings['data_nav_dots'] === 'yes') ? 'true' : 'false';
				$data_autoplay_hover_pause = ($settings['data_autoplay_hover_pause'] === 'yes') ? 'true' : 'false';
				$data_rtl = ($settings['data_rtl'] === 'yes') ? 'true' : 'false'; ?>
					<div class="owl-carousel mwt-posts-carousel" data-number="<?php echo esc_attr($unique_id); ?>" data-rtl="<?php echo esc_attr($data_rtl) ?>" data-loop="<?php echo esc_attr($data_loop) ?>" data-autoplay="<?php echo esc_attr($data_autoplay) ?>" data-autoplaytimeout="<?php echo esc_attr($settings['data_autoplaytimeout']) ?>" data-autoplayhoverpause="<?php echo esc_attr($data_autoplay_hover_pause) ?>" data-center="<?php echo esc_attr($data_center) ?>" data-nav="<?php echo esc_attr($data_nav) ?>" data-dots="<?php echo esc_attr($data_dots) ?>" data-margin="<?php echo esc_attr($settings['data_margin']) ?>" data-responsive-lg="<?php echo esc_attr($settings['data_responsive_lg']) ?>" data-responsive-md="<?php echo esc_attr($settings['data_responsive_md']) ?>" data-responsive-sm="<?php echo esc_attr($settings['data_responsive_sm']) ?>" data-responsive-xs="<?php echo esc_attr($settings['data_responsive_xs']) ?>" data-filters=".filters-<?php echo esc_attr($unique_id); ?>">
					<?php endif;
				//end carousel layout  

				//grid layout
				if ($settings['posts_layout'] === 'grid') :
					$horizontal_order = ($settings['posts_horizontal_order'] === 'yes') ? true : false;
					?>
						<div class="layout-cols layout-cols-<?php echo esc_attr($settings['posts_items_per_row']); ?> layout-gap-<?php echo esc_attr($settings['posts_layout_gap']); ?>">
							<div class="grid-columns-wrapper">
								<div class="masonry" data-filters=".filters-<?php echo esc_attr($unique_id); ?>" data-order="<?php echo esc_attr($horizontal_order); ?>">
									<div class="grid-sizer"></div>
									<?php endif;
								//end grid layout 

								while ($mwt_posts->have_posts()) :
									$mwt_posts->the_post();
									$mwt_post_title = get_the_title($mwt_posts->ID);
									$mwt_title = (!empty($mwt_post_title)) ? $mwt_post_title : esc_html__('(no title)', 'mwt-addons-for-elementor');

									$post_terms = get_the_terms(get_the_ID(), 'category');
									$post_terms_class = '';
									$post_terms_name = '';
									$post_terms_id = '';
									$cat = '';
									$cat_separator = !empty($settings['categories_separator']) ? '<span class="mwt-posts-category-separator">' . $settings['categories_separator'] . ' ' . '</span>' : '';
									foreach ($post_terms as $post_term) {
										$post_terms_class .= $post_term->slug . ' ';
										$post_terms_name = $post_term->name;
										$post_terms_id = $post_term->term_id . ' ';
										$cat .= '<a href="' . esc_url(get_category_link($post_terms_id)) . '" rel="tag">' . $post_terms_name . '' . $cat_separator . '</a>';
									}

									$settings['featured_image'] = array(
										'id' => get_post_thumbnail_id(),
									);
									$thumbnail_html = \Elementor\Group_Control_Image_Size::get_attachment_image_html($settings, 'featured_image');
									if (empty($thumbnail_html)) {
										return;
									}

									//carousel layout
									if ($settings['posts_layout'] === 'carousel') : ?>
										<div class="owl-carousel-item <?php echo esc_attr($post_terms_class); ?>">
										<?php endif;
									//end carousel layout 

									//grid layout
									if ($settings['posts_layout'] === 'grid') :
										$item_layout = !empty($settings['item_layout']) ? $settings['item_layout'] : '';	?>
											<div class="grid-item <?php echo esc_attr($item_layout . ' ' . $post_terms_class); ?>">
											<?php endif;
										//end grid layout  
											?>
											<article <?php post_class('mwt-posts-article'); ?>>
												<?php $show_featured_image = $settings['show_featured_image'];
												if ('yes' === $show_featured_image) : ?>
													<div class="mwt-posts-media-content">
														<a class="mwt-posts-thumbnail" href="<?php the_permalink($mwt_posts->ID); ?>">
															<?php echo wp_kses_post($thumbnail_html); ?>
														</a>
														<?php
														if ('yes' === $settings['show_categories'] && $settings['show_categories_over_thumbnail'] === 'yes') : ?>
															<div class="mwt-posts-categories position-<?php echo esc_attr($settings['categories_position']); ?>">
																<?php echo ($cat); ?>
															</div>
														<?php endif; ?>
														<?php
														if ('yes' === ($settings['show_author_avatar'])) :
															echo '<div class="mwt-posts-author-avatar">';
															$author_id        = get_the_author_meta('ID');
															$custom_image_url = get_the_author_meta('custom_profile_image', $author_id);
															if (!empty($custom_image_url)) {
																echo '<img src="' . esc_url($custom_image_url) . '" alt="' . esc_attr(get_the_author_meta('display_name', $author_id)) . '">';
															} else {
																echo get_avatar($author_id, 60);
															}
															echo '</div>';
														endif; //author_avatar 
														?>
													</div>
												<?php endif ?>
												<div class="mwt-posts-item-content">
													<?php
													$meta_separator = !empty($settings['meta_separator']) ? $settings['meta_separator'] : '';

													$show_categories = $settings['show_categories'];
													$categories_icon = isset($settings['categories_icon']) ? $settings['categories_icon'] : '';
													$text_before_categories = !empty($settings['text_before_categories']) ? $settings['text_before_categories'] : '';

													$show_date = $settings['show_date'];
													$date_icon = isset($settings['date_icon']) ? $settings['date_icon'] : '';
													$text_before_date = !empty($settings['text_before_date']) ? $settings['text_before_date'] : '';

													$show_author = $settings['show_author'];
													$author_icon = isset($settings['author_icon']) ? $settings['author_icon'] : '';
													$text_before_author = !empty($settings['text_before_author']) ? $settings['text_before_author'] : '';

													$show_comments_count = $settings['show_comments_count'];
													$comments_count_icon = isset($settings['comments_count_icon']) ? $settings['comments_count_icon'] : '';
													$text_before_comments_count = !empty($settings['text_before_comments_count']) ? $settings['text_before_comments_count'] : '';

													//footer
													$footer_show_categories = $settings['footer_show_categories'];
													$footer_show_date = $settings['footer_show_date'];
													$footer_show_author = $settings['footer_show_author'];
													$footer_show_comments_count = $settings['footer_show_comments_count'];

													if ('yes' === $show_categories || 'yes' === $show_date || 'yes' === $show_author || 'yes' === $show_comments_count) :
													?>
														<div class="mwt-posts-meta">
															<?php
															if ($show_categories === 'yes' && $footer_show_categories !== 'yes' && $settings['show_categories_over_thumbnail'] !== 'yes') : ?>
																<span class="mwt-posts-meta-item mwt-posts-categories">
																	<?php
																	if ($categories_icon !== '') : ?>
																		<?php \Elementor\Icons_Manager::render_icon($categories_icon, ['aria-hidden' => 'true', 'class' => 'mwt-posts-meta-icon']); ?>
																	<?php endif;
																	if ($text_before_categories !== '') : ?>
																		<span class="mwt-posts-text-before"><?php echo esc_html($text_before_categories); ?></span>
																	<?php endif;
																	echo ($cat);
																	if ($meta_separator !== '') : ?>
																		<span class="mwt-posts-separator"><?php echo esc_html($meta_separator); ?></span>
																	<?php endif; //separator 
																	?>
																</span>
															<?php endif; //categories

															if ($show_date === 'yes' && $footer_show_date !== 'yes') : ?>
																<span class="mwt-posts-meta-item mwt-posts-date">
																	<?php
																	if ($date_icon !== '') : ?>
																		<?php \Elementor\Icons_Manager::render_icon($date_icon, ['aria-hidden' => 'true', 'class' => 'mwt-posts-meta-icon']); ?>
																	<?php endif;
																	if ($text_before_date !== '') : ?>
																		<span class="mwt-posts-text-before"><?php echo esc_html($text_before_date); ?></span>
																	<?php endif; ?>
																	<a href="<?php the_permalink($mwt_posts->ID); ?>"><?php echo get_the_date('', $mwt_posts->ID); ?></a>
																	<?php
																	if ($meta_separator !== '') : ?>
																		<span class="mwt-posts-separator"><?php echo esc_html($meta_separator); ?></span>
																	<?php endif; //separator 
																	?>
																</span>
															<?php endif; //date

															if ($show_author === 'yes' && $footer_show_author !== 'yes') : ?>
																<span class="mwt-posts-meta-item mwt-posts-author">
																	<?php
																	if ($author_icon !== '') : ?>
																		<?php \Elementor\Icons_Manager::render_icon($author_icon, ['aria-hidden' => 'true', 'class' => 'mwt-posts-meta-icon']); ?>
																	<?php endif;
																	if ($text_before_author !== '') : ?>
																		<span class="mwt-posts-text-before"><?php echo esc_html($text_before_author); ?></span>
																	<?php endif; ?>
																	<a href="<?php echo esc_url(get_author_posts_url(get_the_author_meta('ID'))) ?>"><?php echo get_the_author() ?></a>
																	<?php
																	if ($meta_separator !== '') : ?>
																		<span class="mwt-posts-separator"><?php echo esc_html($meta_separator); ?></span>
																	<?php endif; //separator 
																	?>
																</span>
															<?php endif; //author

															if ($show_comments_count === 'yes' && $footer_show_comments_count !== 'yes') : ?>
																<span class="mwt-posts-meta-item mwt-posts-comments-count">
																	<?php
																	if ($comments_count_icon !== '') : ?>
																		<?php \Elementor\Icons_Manager::render_icon($comments_count_icon, ['aria-hidden' => 'true', 'class' => 'mwt-posts-meta-icon']); ?>
																	<?php endif;
																	if ($text_before_comments_count !== '') : ?>
																		<span class="mwt-posts-text-before"><?php echo esc_html($text_before_comments_count); ?></span>
																	<?php endif;
																	if (comments_open()) : ?>
																		<a href="<?php comments_link(); ?>"><?php comments_number("0", "1", "%"); ?></a>
																	<?php else : ?>
																		<span><?php echo esc_html__('Closed', 'mwt-addons-for-elementor'); ?></span>
																	<?php endif; ?>
																</span>
															<?php endif; //comments_count 
															?>
														</div>
													<?php endif; ?>
													<h4 class="mwt-posts-title">
														<a href="<?php the_permalink($mwt_posts->ID); ?>"><?php echo wp_kses_post($mwt_title); ?></a>
													</h4>
													<?php $show_excerpt = $settings['show_excerpt'];
													if ('yes' === $show_excerpt) : ?>
														<div class="mwt-posts-excerpt">
															<?php
															if (!empty($settings['posts_excerpt_length'])) {
																$length = $settings['posts_excerpt_length'];
															} else {
																$length = 55;
															}
															echo wp_kses_post(wp_trim_words(get_the_excerpt($mwt_posts->ID), $length));
															?>
														</div>
													<?php endif ?>
													<?php $show_more_link = $settings['show_more_link'];
													if ('yes' === $show_more_link) : ?>
														<div class="mwt-posts-read-more-btn">
															<a href="<?php the_permalink(); ?>"><?php echo esc_html__('Read More', 'mwt-addons-for-elementor') ?>
																<?php
																$more_link_icon = isset($settings['more_link_icon']) ? $settings['more_link_icon'] : '';
																if ($more_link_icon) : ?>
																	<?php \Elementor\Icons_Manager::render_icon($settings['more_link_icon'], ['aria-hidden' => 'true', 'class' => 'mwt-posts-read-more-btn-icon']); ?>
																<?php endif; ?>
															</a>
														</div>
													<?php endif;
													if ($settings['show_footer'] === 'yes') : ?>
														<footer class="mwt-posts-footer">
															<?php
															$footer_meta_separator = !empty($settings['footer_meta_separator']) ? $settings['footer_meta_separator'] : '';

															if ('yes' === $footer_show_categories && $settings['show_categories_over_thumbnail'] !== 'yes') : ?>
																<span class="mwt-posts-meta-item mwt-posts-categories">
																	<?php
																	if ($categories_icon !== '') : ?>
																		<?php \Elementor\Icons_Manager::render_icon($categories_icon, ['aria-hidden' => 'true', 'class' => 'mwt-posts-meta-icon']); ?>
																	<?php endif;
																	if ($text_before_categories !== '') : ?>
																		<span class="mwt-posts-text-before"><?php echo esc_html($text_before_categories); ?></span>
																	<?php endif;
																	echo ($cat);

																	if ($footer_meta_separator !== '') : ?>
																		<span class="mwt-posts-separator"><?php echo esc_html($footer_meta_separator); ?></span>
																	<?php endif; //separator 
																	?>
																</span>
															<?php endif; //categories

															if ('yes' === $footer_show_date) : ?>
																<span class="mwt-posts-meta-item mwt-posts-date">
																	<?php
																	if ($date_icon !== '') : ?>
																		<?php \Elementor\Icons_Manager::render_icon($date_icon, ['aria-hidden' => 'true', 'class' => 'mwt-posts-meta-icon']); ?>
																	<?php endif;
																	if ($text_before_date !== '') : ?>
																		<span class="mwt-posts-text-before"><?php echo esc_html($text_before_date); ?></span>
																	<?php endif; ?>
																	<a href="<?php the_permalink($mwt_posts->ID); ?>"><?php echo get_the_date('', $mwt_posts->ID); ?></a>
																	<?php

																	if ($footer_meta_separator !== '') : ?>
																		<span class="mwt-posts-separator"><?php echo esc_html($footer_meta_separator); ?></span>
																	<?php endif; //separator 
																	?>
																</span>
															<?php endif; //date

															if ('yes' === $footer_show_author) : ?>
																<span class="mwt-posts-meta-item mwt-posts-author">
																	<?php
																	if ($author_icon !== '') : ?>
																		<?php \Elementor\Icons_Manager::render_icon($author_icon, ['aria-hidden' => 'true', 'class' => 'mwt-posts-meta-icon']); ?>
																	<?php endif;
																	if ($text_before_author !== '') : ?>
																		<span class="mwt-posts-text-before"><?php echo esc_html($text_before_author); ?></span>
																	<?php endif; ?>
																	<a href="<?php echo esc_url(get_author_posts_url(get_the_author_meta('ID'))) ?>"><?php echo get_the_author() ?></a>

																	<?php
																	if ($footer_meta_separator !== '') : ?>
																		<span class="mwt-posts-separator"><?php echo esc_html($footer_meta_separator); ?></span>
																	<?php endif; //separator 
																	?>
																</span>
															<?php endif; //author

															if ('yes' === $footer_show_comments_count) : ?>
																<span class="mwt-posts-meta-item mwt-posts-comments-count">
																	<?php
																	if ($comments_count_icon !== '') : ?>
																		<?php \Elementor\Icons_Manager::render_icon($comments_count_icon, ['aria-hidden' => 'true', 'class' => 'mwt-posts-meta-icon']); ?>
																	<?php endif;
																	if ($text_before_comments_count !== '') : ?>
																		<span class="mwt-posts-text-before"><?php echo esc_html($text_before_comments_count); ?></span>
																	<?php endif;
																	if (comments_open()) : ?>
																		<a href="<?php comments_link(); ?>"><?php comments_number("0", "1", "%"); ?></a>
																	<?php else : ?>
																		<span><?php echo esc_html__('Closed', 'mwt-addons-for-elementor'); ?></span>
																	<?php endif; ?>
																</span>
															<?php endif; //comments_count 
															?>
														</footer>
													<?php endif; ?>
												</div>
											</article>
											</div>

										<?php endwhile; ?>
										<?php wp_reset_postdata(); //reset the query 
										?>
										</div>
										<?php
										//carousel layout 
										if ($settings['posts_layout'] === 'carousel') : ?>
											<div class="owl-nav position-<?php echo esc_attr($settings['prev_next_icons_position']); ?> mwt-posts-carousel-nav-<?php echo esc_attr($unique_id); ?>">
												<?php
												$date_nav_prev = isset($settings['date_nav_prev']) ? $settings['date_nav_prev'] : '';
												$date_nav_next = isset($settings['date_nav_next']) ? $settings['date_nav_next'] : '';
												$date_nav_prev_word = isset($settings['date_nav_prev_word']) ? $settings['date_nav_prev_word'] : '';
												$date_nav_next_word = isset($settings['date_nav_next_word']) ? $settings['date_nav_next_word'] : '';
												?>
												<span class="mwt-posts-carousel-prev-<?php echo esc_attr($unique_id); ?>">
													<?php if ($date_nav_prev) :
														\Elementor\Icons_Manager::render_icon($settings['date_nav_prev'], ['aria-hidden' => 'true', 'class' => 'icon-prev']);
													endif;
													echo esc_html($date_nav_prev_word)
													?>
												</span>
												<span class="mwt-posts-carousel-next-<?php echo esc_attr($unique_id); ?>">
													<?php
													echo esc_html($date_nav_next_word);
													if ($date_nav_next) :
														\Elementor\Icons_Manager::render_icon($settings['date_nav_next'], ['aria-hidden' => 'true', 'class' => 'icon-next']);
													endif;
													?>
												</span>
											</div>
										<?php endif; //end carousel layout 
										?>
								</div>
								<?php
								//grid layout 
								if ($settings['posts_layout'] === 'grid') : ?>
							</div>
						</div>
			<?php endif; //end grid layout 
							}
						} ?>