<?php

/**
 * Template helpers fucntions
 *
 * @package Energy
 * @since 0.0.1
 */

if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

if (!function_exists('energy_get_body_schema_itemtype')) :

	/**
	 * Get itemtype value for body tag
	 *
	 * @since 0.0.1
	 */
	function energy_get_body_schema_itemtype()
	{

		//Get default itemtype
		$energy_itemtype = (is_page()) ? 'WebPage' : 'Blog';

		//Change itemtype if is search page
		if (is_search()) {
			$energy_itemtype = 'SearchResultsPage';
		}

		return $energy_itemtype;
	}
endif;

//get theme template part
if (!function_exists('energy_template_part')) :
	function energy_template_part($energy_template_part_name, $energy_default_value = '1')
	{
		$energy_return = energy_option($energy_template_part_name, $energy_default_value);

		//for demo
		if (!empty($_GET[$energy_template_part_name])) {
			$energy_return = absint($_GET[$energy_template_part_name]);
		}

		return $energy_return;
	}
endif;

//get proper CSS classes for #main section based on page template
if (!function_exists('energy_get_page_main_section_css_classes')) :
	function energy_get_page_main_section_css_classes()
	{
		$return = 'width-inherit';
		if (is_page_template('page-templates/no-sidebar-720.php')) {
			$return = 'container-720';
		}
		if (is_page_template('page-templates/no-sidebar-960.php')) {
			$return = 'container-960';
		}
		if (is_page_template('page-templates/no-sidebar-1170.php')) {
			$return = 'container-1170';
		}

		return $return;
	}
endif;

//get proper CSS classes for main column, aside column and body
if (!function_exists('energy_get_layout_css_classes')) :
	function energy_get_layout_css_classes()
	{

		//default - sidebar
		$energy_return = array(
			'body'  => 'with-sidebar',
			'main'  => 'column-main',
			'aside' => 'column-aside',
		);

		//check for shop
		if (function_exists('is_woocommerce') && is_woocommerce()) :
			if (is_product()) {
				$energy_shop_sidebar_position_option = energy_option('product_sidebar_position', 'right');
			}
			if (is_shop() || is_product_taxonomy()) {
				$energy_shop_sidebar_position_option = energy_option('shop_sidebar_position', 'right');
			}
			//if empty sidebar or disabled in customizer - removing aside
			if (!is_active_sidebar('shop') || 'no' === $energy_shop_sidebar_position_option) {
				$energy_return['body']  = 'no-sidebar';
				$energy_return['aside'] = false;

				return $energy_return;
			} //is_active_sidebar( 'shop' )
			//left sidebar
			if ('left' === $energy_shop_sidebar_position_option) {
				$energy_return['body'] .= ' sidebar-left';

				return $energy_return;
				//default - right sidebar
			} else {
				return $energy_return;
			}
		endif; //is_woocommerce()

		//check for shop wishlist
		if (function_exists('yith_wcwl_is_wishlist_page')) {
			if (yith_wcwl_is_wishlist_page()) {
				$energy_shop_sidebar_position_option = energy_option('shop_sidebar_position', 'right');
				//if empty sidebar or disabled in customizer - removing aside
				if (!is_active_sidebar('shop') || 'no' === $energy_shop_sidebar_position_option) {
					$energy_return['body']  = 'no-sidebar';
					$energy_return['aside'] = false;

					return $energy_return;
				}
				//left sidebar
				if ('left' === $energy_shop_sidebar_position_option) {
					$energy_return['body'] .= ' sidebar-left';

					return $energy_return;
					//default - right sidebar
				} else {
					return $energy_return;
				}
			}
		}; //yith_wcwl_is_wishlist_page()

		//if category has meta - overriding default customizer option option
		if (is_category()) {
			$energy_sidebar_position_option = energy_get_category_sidebar_position();
		} else {
			if (!is_single()) {
				$energy_sidebar_position_option = energy_option('blog_sidebar_position', 'right');
			} else {
				$energy_sidebar_position_option = energy_option('blog_single_sidebar_position', 'right');
			}
		} //is_category

		if (!is_page_template('page-templates/home.php')) {
			$sidebar = 'sidebar-1';
			if (is_front_page() && is_active_sidebar('sidebar-home-main')) {
				$sidebar = 'sidebar-home-main';
			}
			//if empty sidebar - removing aside
			if (!is_active_sidebar($sidebar)) {
				$energy_return['body']  = 'no-sidebar';
				$energy_return['aside'] = false;

				return $energy_return;
			} //sidebar-1
		} else {
			//if empty sidebar on home.php page template - removing aside
			if (!is_active_sidebar('sidebar-home-main')) {
				$energy_return['body']  = 'no-sidebar';
				$energy_return['aside'] = false;

				return $energy_return;
			} //sidebar-nome-main
		} //! is_page_template( 'page-templates/home.php'

		//various cases with sidebar
		//single post without sidebars
		if (is_single()) {

			//no sidebar for posts layouts
			if (
				is_page_template('page-templates/post-full-width-no-meta-no-thumbnail.php')
				||
				is_page_template('page-templates/post-full-width-no-meta.php')
				||
				is_page_template('page-templates/post-full-width.php')
			) {
				$energy_return['body']  = 'no-sidebar';
				$energy_return['aside'] = false;

				return $energy_return;
			}
		} //is_single

		//pages
		if (is_page()) :

			//no sidebar
			if (
				is_page_template('page-templates/full-width.php')
				||
				is_page_template('page-templates/empty-page.php')
				||
				is_page_template('page-templates/empty-page-container.php')
				||
				is_page_template('page-templates/no-sidebar-720.php')
				||
				is_page_template('page-templates/no-sidebar-960.php')
				||
				is_page_template('page-templates/no-sidebar-1170.php')
				||
				is_page_template('page-templates/no-sidebar-no-title.php')
				||
				is_page_template('page-templates/no-sidebar-no-padding.php')
				||
				!is_page_template()
			) {
				$energy_return['body']  = 'no-sidebar';
				$energy_return['aside'] = false;

				return $energy_return;
			}

			//left sidebar for page
			if (
				is_page_template('page-templates/sidebar-left.php')
				||
				('left' === $energy_sidebar_position_option)
			) {
				$energy_return['body'] .= ' sidebar-left';

				return $energy_return;
			}

		//right sidebar is default
		endif; // is_page

		//if no sidebar option - removing aside
		if ('no' === $energy_sidebar_position_option && !(is_page_template('page-templates/home.php'))) {
			$energy_return['body']  = 'no-sidebar';
			$energy_return['aside'] = false;

			return $energy_return;
		}

		//left sidebar
		if ('left' === $energy_sidebar_position_option) {
			$energy_return['body'] .= ' sidebar-left';
		}

		return $energy_return;
	}
endif;

//get category layout based on category meta with global blog option as fallback
if (!function_exists('energy_get_category_layout')) :
	function energy_get_category_layout()
	{
		$energy_layout = '';

		$energy_queried_object = get_queried_object();
		$energy_term_id        = $energy_queried_object->term_id;

		//if layout is overriden for category in admin panel
		$energy_term_metas_layout = get_term_meta($energy_term_id, 'layout', true);
		if (!empty($energy_term_metas_layout)) {
			$energy_layout = $energy_term_metas_layout;
		}

		//if category layout not specified - getting default layout
		if (empty($energy_layout)) {
			$energy_layout = energy_option('blog_layout', '') ? energy_option('blog_layout', '') : 'default';
		}

		return $energy_layout;
	}
endif;

//get category layout gap based on category meta with global blog option as fallback
if (!function_exists('energy_get_category_layout_gap')) :
	function energy_get_category_layout_gap()
	{
		$energy_layout_gap = '';

		$energy_queried_object = get_queried_object();
		$energy_term_id        = $energy_queried_object->term_id;

		//if layout is overriden for category in admin panel
		$energy_term_metas_layout = get_term_meta($energy_term_id, 'gap', true);
		if (!empty($energy_term_metas_layout)) {
			$energy_layout_gap = $energy_term_metas_layout;
		}

		//if category layout not specified - getting default layout
		if (empty($energy_layout_gap)) {
			$energy_layout_gap = energy_option('blog_layout_gap', '') ? energy_option('blog_layout_gap', '') : '';
		}

		return $energy_layout_gap;
	}
endif;

//get feed shot_title
if (!function_exists('energy_get_feed_shot_title')) :
	function energy_get_feed_shot_title()
	{
		if (is_category()) {
			$energy_show_title = !energy_option('title_show_title', '');
		} else {
			$energy_show_title = !energy_option('title_show_title', '') && !is_front_page();
		}

		return $energy_show_title;
	}
endif;

//get feed layout
if (!function_exists('energy_get_feed_layout')) :
	function energy_get_feed_layout()
	{
		if (is_category()) {
			$energy_layout = energy_get_category_layout();
		} else {
			$energy_layout = energy_option('blog_layout', '') ? energy_option('blog_layout', '') : 'default';
		}

		//override option for demo purposes
		if (isset($_GET['blog_layout'])) {
			$energy_layout_id = absint($_GET['blog_layout']);
			$energy_layouts   = array_keys(energy_get_feed_layout_options());
			$energy_layout    = !empty($energy_layouts[$energy_layout_id]) ? $energy_layouts[$energy_layout_id] : $energy_layout;
		}

		return $energy_layout;
	}
endif;

//get feed gap
if (!function_exists('energy_get_feed_gap')) :
	function energy_get_feed_gap()
	{
		if (is_category()) {
			$energy_layout_gap = energy_get_category_layout_gap();
		} else {
			$energy_layout_gap = energy_option('blog_layout_gap', '') ? energy_option('blog_layout_gap', '') : '';
		}

		//override option for demo purposes
		if (isset($_GET['blog_layout_gap'])) {
			$energy_layout_gap_id = absint($_GET['blog_layout_gap']);
			$energy_layout_gaps   = array_keys(energy_get_feed_layout_gap_options());
			$energy_layout_gap    = !empty($energy_layout_gaps[$energy_layout_gap_id]) ? $energy_layout_gaps[$energy_layout_gap_id] : $energy_layout_gap;
		}

		return $energy_layout_gap;
	}
endif;

//get category sidebar_position based on category meta with global blog option as fallback
if (!function_exists('energy_get_category_sidebar_position')) :
	function energy_get_category_sidebar_position()
	{
		$energy_sidebar_position = '';

		$energy_queried_object = get_queried_object();
		$energy_term_id        = $energy_queried_object->term_id;

		//term metas from category options has higher priority than customizer option for special categories
		$energy_term_metas = get_term_meta($energy_term_id, 'sidebar_position', true);
		if (!empty($energy_term_metas)) {
			$energy_sidebar_position = $energy_term_metas;
		}

		//if category sidebar_position not specified - getting default sidebar_position
		if (empty($energy_sidebar_position)) {
			$energy_sidebar_position = energy_option('blog_sidebar_position', '') ? energy_option('blog_sidebar_position', '') : 'right';
		}

		return $energy_sidebar_position;
	}
endif;

//get single post layout based on blog post option
if (!function_exists('energy_get_post_layout')) :
	function energy_get_post_layout()
	{

		$energy_layout = energy_option('blog_single_layout', '') ? energy_option('blog_single_layout', '') : 'default';

		//override option for demo purposes
		if (isset($_GET['blog_single_layout'])) {
			$energy_layout_id = absint($_GET['blog_single_layout']);
			$energy_layouts   = array_keys(energy_get_post_layout_options());
			$energy_layout    = !empty($energy_layouts[$energy_layout_id]) ? $energy_layouts[$energy_layout_id] : $energy_layout;
		}

		return $energy_layout;
	}
endif;

if (!function_exists('energy_body_classes')) :
	/**
	 * Adds custom classes to the array of body classes.
	 *
	 * @param array $energy_classes Classes for the body element.
	 *
	 * @return array
	 */
	function energy_body_classes($energy_classes)
	{
		//header-empty
		if (is_page_template('page-templates/empty-page.php')) {
			$energy_classes[] = 'header-empty';
		}

		//header sticky
		if (has_nav_menu('side') || is_active_sidebar('sidebar-side')) {
			$energy_classes[] = 'has-side-nav';
			$energy_classes[] = energy_option('side_nav_position', '') ? 'side-nav-right' : 'side-nav-left';
			$energy_classes[] = energy_option('side_nav_sticked', '') ? 'side-nav-sticked' : '';
			$energy_classes[] = energy_option('side_nav_header_overlap', '') ? 'side-nav-header-overlap' : '';
		}

		// Adds a class of hfeed to non-singular pages.
		if (!is_singular()) {
			$energy_classes[] = 'hfeed';
		} else {
			//add 'singular' class for single post or page or any other post type
			$energy_classes[] = 'singular';
		}

		//Adds a sidebar classes
		$energy_css_classes = energy_get_layout_css_classes();

		$energy_classes[] = $energy_css_classes['body'];

		//Add icons in meta classes
		//single post
		if (is_singular()) {
			$energy_hide_meta_icons = energy_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$energy_hide_meta_icons = energy_option('blog_hide_meta_icons', false);
		}
		if ($energy_hide_meta_icons) {
			$energy_classes[] = 'meta-icons-hidden';
		}

		$blog_single_hide_meta_categories_icons = energy_option('blog_single_hide_meta_categories_icons', false);
		if (is_singular() && !empty($blog_single_hide_meta_categories_icons)) {
			$energy_classes[] = 'meta-icons-categories-hidden';
		}

		//container width
		$energy_container_width      = energy_option('main_container_width', '1170');
		$energy_container_post_width = energy_option('blog_single_container_width', '');
		$energy_container_blog_width = energy_option('blog_container_width', '');
		if (is_singular('post') && !empty($energy_container_post_width)) {
			$energy_container_width = $energy_container_post_width;
		}
		if ((is_home() || is_category() || is_tag() || is_date() || is_author()) && !empty($energy_container_blog_width)) {
			$energy_container_width = $energy_container_blog_width;
		}
		if ('1520' === $energy_container_width) {
			$energy_classes[] = 'container-1520';
		}
		if ('1170' === $energy_container_width) {
			$energy_classes[] = 'container-1170';
		}
		if ('960' === $energy_container_width) {
			$energy_classes[] = 'container-960';
		}
		if ('720' === $energy_container_width) {
			$energy_classes[] = 'container-720';
		}

		//meta icons color class
		$energy_meta_icons_color = energy_option('color_meta_icons', '');
		if ($energy_meta_icons_color) {
			$energy_classes[] = esc_attr($energy_meta_icons_color);
		}

		//shop class
		if (class_exists('WooCommerce')) {
			$energy_classes[] = 'woo';
		}

		//header class
		if ('always-sticky' === energy_option('header_sticky', '')) {
			$energy_classes[] = 'header-sticky';
		}

		//animation enabled
		$energy_animation = energy_option('animation_enabled', '');
		if (!empty($energy_animation) && !is_customize_preview()) {
			$energy_classes[] = 'animation-enabled';
		}

		//title section enabled
		$energy_title = energy_is_title_section_is_shown();
		if (empty($energy_title)) {
			$energy_classes[] = 'title-hidden';
		}

		return $energy_classes;
	}
endif;
add_filter('body_class', 'energy_body_classes');

//markup for animated page elements
if (!function_exists('energy_animated_elements_markup')) :
	function energy_animated_elements_markup()
	{
		$energy_animation = energy_option('animation_enabled', '');
		if (empty($energy_animation)) {
			return;
		}

		//get animations array from customizer. Keys - selectors
		$energy_animations = array(
			'.column-aside .widget'            => energy_option('animation_sidebar_widgets', ''),
			'.footer-top-widgets .widget'      => energy_option('animation_footer_top_widgets', ''),
			'.footer-widgets .widget'          => energy_option('animation_footer_widgets', ''),
			'.hfeed article.post'              => energy_option('animation_feed_posts', ''),
			'.hfeed .post .post-thumbnail img' => energy_option('animation_feed_posts_thumbnail', ''),
		);

		$energy_animations = array_filter($energy_animations);
		if (!empty($energy_animations) && !is_customize_preview()) :
?>
			data-animate='<?php echo esc_attr(str_replace('&quot;', '"', json_encode($energy_animations))); ?>'
		<?php
		endif;
	}
endif;

//markup for sticky post label
if (!function_exists('energy_sticky_post_label')) :
	function energy_sticky_post_label()
	{
		if (is_sticky() && is_home() && !is_paged()) :
		?>
			<span class="icon-inline sticky-post">
				<?php energy_icon('thumb-tack'); ?>
			</span><!-- .sticky-post -->
			<?php
		endif; //is_sticky()
	}
endif;

//arguments for link pages
if (!function_exists('energy_get_wp_link_pages_atts')) :
	function energy_get_wp_link_pages_atts()
	{
		return apply_filters(
			'energy_link_pages_atts',
			array(
				'before'      => '<div class="page-links"><span class="screen-reader-text">' . esc_html__('Pages: ', 'energy') . '</span>',
				'after'       => '</div>',
				'link_before' => '<span>',
				'link_after'  => '</span>',
			)
		);
	}
endif;

//arguments for link pages
if (!function_exists('energy_get_the_posts_pagination_atts')) :
	function energy_get_the_posts_pagination_atts()
	{
		return array(
			'mid_size'  => 5,
			'prev_text' => '<span class="screen-reader-text">' . esc_html__('Previous page', 'energy') . '</span><span class="icon-inline">' . energy_icon('chevron-left', true) . '</span>',
			'next_text' => '<span class="screen-reader-text">' . esc_html__('Next page', 'energy') . '</span><span class="icon-inline">' . energy_icon('chevron-right', true) . '</span>',
		);
	}
endif;

//get top level menu items count
if (!function_exists('energy_get_menu_top_level_items_count')) :
	function energy_get_menu_top_level_items_count($energy_menu_name)
	{

		$energy_locations   = get_nav_menu_locations();
		$energy_menu_id     = $energy_locations[$energy_menu_name];
		$energy_menu_object = wp_get_nav_menu_object($energy_menu_id);
		if (empty($energy_menu_object)) {
			return '-1';
		}

		$energy_menu_items       = wp_get_nav_menu_items($energy_menu_object->term_id);
		$energy_menu_items_count = 0;

		foreach ((array) $energy_menu_items as $energy_key => $energy_menu_item) {
			if ('0' === $energy_menu_item->menu_item_parent) {
				$energy_menu_items_count++;
			}
		}

		return $energy_menu_items_count;
	}
endif;

//get menu class depending on menu top level items count
if (!function_exists('energy_get_menu_class_based_on_top_items_count')) :
	function energy_get_menu_class_based_on_top_items_count($energy_menu_name)
	{

		$energy_menu_items_count = energy_get_menu_top_level_items_count($energy_menu_name);
		if ('-1' === $energy_menu_items_count) {
			return 'menu-empty';
		}

		$energy_css_class = 'menu-low-items';

		if ($energy_menu_items_count > 5) {
			$energy_css_class = 'menu-many-items';
		}

		return $energy_css_class;
	}
endif;


//print svg icon
if (!function_exists('energy_icon')) :
	function energy_icon($energy_name, $energy_return = false, $energy_container_css_class = 'svg-icon')
	{
		//in the future we'll add option for this
		$energy_icons_pack = 'google';

		if ($energy_return) {
			ob_start();
		}

		echo '<span class="' . esc_attr($energy_container_css_class) . ' icon-' . esc_attr($energy_name) . '">';
		get_template_part('/template-parts/svg/' . $energy_icons_pack . '/' . $energy_name);
		echo '</span>';

		if ($energy_return) {
			return ob_get_clean();
		}
	}
endif;

//print social link
if (!function_exists('energy_social_link')) :
	function energy_social_link($energy_name, $energy_url)
	{
		echo '<a href="' . esc_url($energy_url) . '" target="_blank" class="social-icon social-icon-' . esc_attr($energy_name) . '">';
		energy_icon($energy_name);
		echo '</a>';
	}
endif;

//meta
//get meta array
if (!function_exists('energy_get_theme_meta')) :
	function energy_get_theme_meta($energy_meta_names = array())
	{
		/*
		customizer options with meta are:
			'meta_email'
			'meta_email_label'
			'meta_email_2'
			'meta_email_2_label'
			'meta_phone'
			'meta_phone_label'
			'meta_phone_2'
			'meta_phone_label_2'
			'meta_address'
			'meta_address_label'
			'meta_opening_hours'
			'meta_opening_hours_label'
		*/

		//if no names specified - using all meta
		if (empty($energy_meta_names)) :
			$energy_meta_names = array(
				'email',
				'email_2',
				'phone',
				'phone_2',
				'address',
				'opening_hours',
			);
		endif;

		$energy_theme_meta = array();

		//meta values
		foreach ($energy_meta_names as $energy_meta_name) {
			$energy_value = energy_option('meta_' . $energy_meta_name);
			if (!empty($energy_value)) {
				$energy_theme_meta[$energy_meta_name] = $energy_value;
			}
		}

		//labels for meta if it is not empty
		if (!empty($energy_theme_meta)) {
			foreach ($energy_theme_meta as $energy_meta_name => $energy_meta_value) {
				$energy_label = energy_option('meta_' . $energy_meta_name . '_label');
				if (!empty($energy_label)) {
					$energy_theme_meta[$energy_meta_name . '_label'] = $energy_label;
				}
			}
		}

		return $energy_theme_meta;
	}
endif;

//print all social links based on theme_meta from Customizzer
if (!function_exists('energy_social_links')) :
	function energy_social_links($wrapper_class = '', $links_class = '')
	{

		$energy_facebook  = energy_option('meta_facebook');
		$energy_twitter   = energy_option('meta_twitter');
		$energy_linkedin  = energy_option('meta_linkedin');
		$energy_youtube   = energy_option('meta_youtube');
		$energy_instagram = energy_option('meta_instagram');
		$energy_pinterest = energy_option('meta_pinterest');
		$energy_github    = energy_option('meta_github');

		if (
			!empty($energy_facebook)
			||
			!empty($energy_twitter)
			||
			!empty($energy_youtube)
			||
			!empty($energy_instagram)
			||
			!empty($energy_pinterest)
			||
			!empty($energy_linkedin)
			||
			!empty($energy_github)
		) :

			if ($wrapper_class) {
				echo '<div class="' . esc_attr($wrapper_class) . '">';
			}
			echo '<span class="social-links' . ' ' . $links_class . '">';

			if (!empty($energy_facebook)) :
				energy_social_link('facebook', $energy_facebook);
			endif;

			if (!empty($energy_twitter)) :
				energy_social_link('twitter', $energy_twitter);
			endif;

			if (!empty($energy_linkedin)) :
				energy_social_link('linkedin', $energy_linkedin);
			endif;

			if (!empty($energy_youtube)) :
				energy_social_link('youtube', $energy_youtube);
			endif;

			if (!empty($energy_instagram)) :
				energy_social_link('instagram', $energy_instagram);
			endif;

			if (!empty($energy_pinterest)) :
				energy_social_link('pinterest', $energy_pinterest);
			endif;

			if (!empty($energy_github)) :
				energy_social_link('github-circle', $energy_github);
			endif;

			echo '</span><!--.social-links-->';
			if ($wrapper_class) {
				echo '</div><!--' . esc_html($wrapper_class) . '-->';
			}

		endif;
	}
endif;

//print copyright social link
if (!function_exists('energy_social_link_2')) :
	function energy_social_link_2($energy_name, $energy_url)
	{
		echo '<a href="' . esc_url($energy_url) . '" target="_blank" class="social-icon social-icon-' . esc_attr($energy_name) . '">';
		energy_icon($energy_name);
		echo '</a>';
	}
endif;

//print all social links based on theme_meta from Customizzer 
if (!function_exists('energy_social_links_copyright')) :
	function energy_social_links_copyright($wrapper_class = '')
	{

		$energy_twitter   = energy_option('meta_twitter');
		$energy_facebook  = energy_option('meta_facebook');
		$energy_linkedin  = energy_option('meta_linkedin');
		$energy_youtube   = energy_option('meta_youtube');
		$energy_instagram = energy_option('meta_instagram');
		$energy_pinterest = energy_option('meta_pinterest');
		$energy_github    = energy_option('meta_github');

		if (
			!empty($energy_facebook)
			||
			!empty($energy_twitter)
			||
			!empty($energy_youtube)
			||
			!empty($energy_instagram)
			||
			!empty($energy_pinterest)
			||
			!empty($energy_linkedin)
			||
			!empty($energy_github)
		) :

			if ($wrapper_class) {
				echo '<div class="' . esc_attr($wrapper_class) . '">';
			}
			echo '<span class="social-links copyright-social-links divided-content">';

			if (!empty($energy_facebook)) :
				energy_social_link_2('facebook', $energy_facebook);
			endif;

			if (!empty($energy_twitter)) :
				energy_social_link_2('twitter', $energy_twitter);
			endif;

			if (!empty($energy_youtube)) :
				energy_social_link_2('youtube', $energy_youtube);
			endif;

			if (!empty($energy_instagram)) :
				energy_social_link_2('instagram', $energy_instagram);
			endif;

			if (!empty($energy_pinterest)) :
				energy_social_link_2('pinterest', $energy_pinterest);
			endif;

			if (!empty($energy_linkedin)) :
				energy_social_link_2('linkedin', $energy_linkedin);
			endif;

			if (!empty($energy_github)) :
				energy_social_link_2('github-circle', $energy_github);
			endif;

			echo '</span><!--.social-links-->';
			if ($wrapper_class) {
				echo '</div><!--' . esc_html($wrapper_class) . '-->';
			}

		endif;
	}
endif;

//print meta link
if (!function_exists('energy_link_theme_meta_widget')) :
	function energy_link_theme_meta_widget($energy_name, $energy_url)
	{
		echo '<a href="' . esc_url($energy_url) . '" target="_blank" class="meta-icon border-icon meta-icon-' . esc_attr($energy_name) . '">';
		energy_icon($energy_name);
		echo '</a>';
	}
endif;

//print social link
if (!function_exists('energy_social_link_theme_meta_widget')) :
	function energy_social_link_theme_meta_widget($energy_name, $energy_url)
	{
		echo '<a href="' . esc_url($energy_url) . '" target="_blank" class="social-icon border-icon social-icon-' . esc_attr($energy_name) . '">';
		energy_icon($energy_name);
		echo '</a>';
	}
endif;

//print all social links based on theme_meta from Customizzer
if (!function_exists('energy_social_links_theme_meta_widget')) :
	function energy_social_links_theme_meta_widget($wrapper_class = '')
	{

		$energy_twitter   = energy_option('meta_twitter');
		$energy_facebook  = energy_option('meta_facebook');
		$energy_linkedin  = energy_option('meta_linkedin');
		$energy_youtube   = energy_option('meta_youtube');
		$energy_instagram = energy_option('meta_instagram');
		$energy_pinterest = energy_option('meta_pinterest');
		$energy_github    = energy_option('meta_github');

		if (
			!empty($energy_facebook)
			||
			!empty($energy_twitter)
			||
			!empty($energy_youtube)
			||
			!empty($energy_instagram)
			||
			!empty($energy_pinterest)
			||
			!empty($energy_linkedin)
			||
			!empty($energy_github)
		) :

			if ($wrapper_class) {
				echo '<div class="' . esc_attr($wrapper_class) . '">';
			}
			echo '<span class="social-links">';

			if (!empty($energy_twitter)) :
				energy_social_link_theme_meta_widget('twitter', $energy_twitter);
			endif;

			if (!empty($energy_facebook)) :
				energy_social_link_theme_meta_widget('facebook', $energy_facebook);
			endif;

			if (!empty($energy_instagram)) :
				energy_social_link_theme_meta_widget('instagram', $energy_instagram);
			endif;

			if (!empty($energy_youtube)) :
				energy_social_link_theme_meta_widget('youtube', $energy_youtube);
			endif;

			if (!empty($energy_pinterest)) :
				energy_social_link_theme_meta_widget('pinterest', $energy_pinterest);
			endif;

			if (!empty($energy_linkedin)) :
				energy_social_link_theme_meta_widget('linkedin', $energy_linkedin);
			endif;

			if (!empty($energy_github)) :
				energy_social_link_theme_meta_widget('github-circle', $energy_github);
			endif;

			echo '</span><!--.social-links-->';
			if ($wrapper_class) {
				echo '</div><!--' . esc_html($wrapper_class) . '-->';
			}

		endif;
	}
endif;

if (!function_exists('energy_has_post_thumbnail')) :
	/**
	 * Check if has post thumbnail and thumbnail file exists
	 */
	function energy_has_post_thumbnail($energy_id = false)
	{
		if (empty($energy_id)) {
			$energy_id = get_the_ID();
		}

		return !(post_password_required($energy_id)
			||
			is_attachment()
			||
			!has_post_thumbnail($energy_id)
			||
			!file_exists(get_attached_file(get_post_thumbnail_id($energy_id))));
	}
endif;

if (!function_exists('energy_post_thumbnail')) :
	/**
	 * Displays an optional post thumbnail.
	 *
	 * Wraps the post thumbnail in an anchor element on index views, or a div
	 * element when on single views.
	 */
	function energy_post_thumbnail($energy_size = 'energy-default-post', $energy_css_class = '', $show_categories = false)
	{

		if (
			!energy_has_post_thumbnail()
		) {
			return;
		}

		//detect video
		$oembed_url = false;
		$oembed_post_thumbnail = false;
		//only video post format
		if ('video' === get_post_format()) {
			$oembed_post_thumbnail = true;
		}

		if ($oembed_post_thumbnail) {
			$post_content = get_the_content();
			//get oEmbed URL
			$reg = preg_match('|^\s*(https?://[^\s"]+)\s*$|im', $post_content, $matches);

			$oembed_url = !empty($reg) ? trim($matches[0]) : false;
			//if no youtube, trying to find self hosted

			$first_self_hosted = '';
			$embeds = array();
			if (empty($oembed_url)) {
				$post_content = apply_filters('the_content', $post_content);
				$embeds = get_media_embedded_in_content($post_content);
			}
		}

		if (is_singular()) :
			if (($oembed_url || !empty($embeds[0]))) :
				//if youtube
				if ($oembed_url) :
					add_filter('the_content', function ($content) use ($oembed_url) {
						//remove embed
						$content = str_replace($oembed_url, '', $content);
						//hide embed wrapper
						$pos = strpos($content, 'class="wp-block-embed');
						if ($pos !== false) {
							$content = substr_replace($content, 'class="d-none wp-block-embed', $pos, strlen('class="wp-block-embed'));
						}
						return $content;
						//1 - to run early
					}, 1);
			?>
					<figure class="wp-block-embed wp-embed-aspect-16-9 post-thumbnail mb-0">
						<div class="wp-block-embed__wrapper" itemprop="video" itemscope="itemscope" itemtype="https://schema.org/VideoObject">
							<?php

							echo wp_oembed_get($oembed_url);

							$thumbnail_url = get_the_post_thumbnail_url(get_the_ID(), 'full');

							if ($thumbnail_url) :
							?>
								<meta itemprop="thumbnailUrl" content="<?php echo esc_url($thumbnail_url); ?>">
							<?php
							endif; //thumbnail_url
							?>
							<meta itemprop="uploadDate" content="<?php echo esc_attr(the_time(get_option('date_format'))); ?>">
							<meta itemprop="contentUrl" content="<?php echo esc_url($oembed_url); ?>">
							<?php
							the_title('<h3 class="d-none" itemprop="name">', '</h3>');
							?>
							<p class="d-none" itemprop="description">
								<?php echo wp_kses(get_the_excerpt(), false); ?>
							</p>
						</div>
					</figure><!-- .post-thumbnail -->
				<?php
				//self hosted
				else :
					$embed = (!empty($embeds[0])) ? $embeds[0] : false;
					$url = preg_match('`src="(.*)"`', $embed, $founds);
					$hosted_video_url = !empty($founds['1']) ? $founds['1'] : '';
					add_filter('the_content', function ($content) use ($embed) {
						//remove embed
						$content = str_replace($embed, '', $content);
						//hide embed wrapper
						$pos = strpos($content, 'class="wp-block-video');
						if ($pos !== false) {
							$content = substr_replace($content, 'class="d-none wp-block-video', $pos, strlen('class="wp-block-embed'));
						}
						return $content;
						//1 - to run early
					}, 1);
				?>
					<figure class="post-thumbnail mb-0">
						<div class="wp-block-video" itemprop="video" itemscope="itemscope" itemtype="https://schema.org/VideoObject">
							<?php
							echo wp_kses_post($embed);
							$thumbnail_url = get_the_post_thumbnail_url(get_the_ID(), 'full');
							if ($thumbnail_url) :
							?>
								<meta itemprop="thumbnailUrl" content="<?php echo esc_url($thumbnail_url); ?>">
							<?php
							endif; //thumbnail_url
							?>
							<meta itemprop="uploadDate" content="<?php echo esc_attr(the_time(get_option('date_format'))); ?>">
							<meta itemprop="contentUrl" content="<?php echo esc_url($hosted_video_url); ?>">
							<?php
							the_title('<h3 class="d-none" itemprop="name">', '</h3>');
							?>
							<p class="d-none" itemprop="description">
								<?php echo wp_kses(get_the_excerpt(), false); ?>
							</p>
						</div>
					</figure><!-- .post-thumbnail -->
				<?php
				endif; //$oembed_url
			//not video
			else :
				?>
				<figure class="<?php echo esc_attr('post-thumbnail ' . $energy_css_class); ?>">
					<?php
					if ($show_categories) {
						energy_entry_meta(false, false, true, false, false);
					}
					the_post_thumbnail(
						$energy_size,
						array(
							'itemprop' => 'image',
							'alt'      => get_the_title(),
						)
					);
					?>
				</figure><!-- .post-thumbnail -->
			<?php
			//oembed
			endif;
		//not is_singular
		//archive
		else :

			//detecting gallery
			$is_gallery = false;
			$gallery_css_class = '';
			$image_size = (energy_option('blog_layout') === 'grid') ? 'energy-square' : 'energy-default-post';
			if ('gallery' === get_post_format()) {
				$galleries_images = get_post_galleries_images();
				//gutenberg block parse
				if (!empty($galleries_images)) {
					global $post;
					if (has_block('gallery', $post->post_content)) {
						$post_blocks = parse_blocks($post->post_content);
						foreach ($post_blocks as $post_block) {
							if ('core/gallery' === $post_block['blockName']) {
								$src_array = array();
								$gallery_imgs_ids = [];
								foreach ($post_block['innerBlocks'] as $subel) {
									$gallery_imgs_ids[] = $subel['attrs']['id'];
								}
								foreach ($gallery_imgs_ids as $id) {
									$src_array[] = wp_get_attachment_image_url($id, $image_size);
								}
								$galleries_images = $src_array;
								break;
							}
						}
					}
				}

				$galleries_images_count = count($galleries_images);
				if ($galleries_images_count) {
					$is_gallery = true;
					$gallery_css_class = 'item-media-gallery';
				}
			} //gallery post format

			?>
			<div class="post-thumbnail-wrap">
				<?php
				if ($show_categories) {
					energy_entry_meta(false, false, true, false, false, false, false);
				}
				?>
				<figure class="<?php echo esc_attr('post-thumbnail ' . $energy_css_class); ?>">
					<a href="<?php

								if (empty($oembed_url)) {
									the_permalink();
								} else {
									echo esc_url($oembed_url);
								}

								?>">
						<?php

						if (empty($is_gallery)) {


							the_post_thumbnail(
								$energy_size,
								array(
									'itemprop' => 'image',
									'alt'      => get_the_title(),
								)
							);
						} else {
							//gallery

							echo '<div class="flexslider"><ul class="slides">';

							//adding featured image as a first element in carousel

							//featured image url
							$post_featured_image_src = wp_get_attachment_image_url(get_post_thumbnail_id(), $image_size);

							if ($post_featured_image_src) : ?>
								<li><img src="<?php echo esc_url($post_featured_image_src); ?>" alt="<?php echo esc_attr(get_the_title()); ?>">
								</li>
							<?php endif;
							$count = 1;

							foreach ($galleries_images as $gallerie) :
								//foreach ($gallerie as $src) :
								//showing only 3 images from gallery
								if ($count > 9) {
									break;
								}
							?>
								<li><img src="<?php echo esc_url($gallerie); ?>" alt="<?php echo esc_attr(get_the_title()); ?>">
								</li>
						<?php
								$count++;
							//endforeach;
							endforeach;
							echo '</ul></div><!--.flexslider-->';
						}
						energy_post_format_icon(get_post_format());
						?>
					</a>
				</figure>
			</div>
			<!-- post-thumbnail-wrap -->
		<?php
		endif; // End is_singular().
	}
endif; //energy_post_thumbnail

//print post format icon
if (!function_exists('energy_post_format_icon')) :
	function energy_post_format_icon($energy_post_format = '')
	{
		// 'video', 'audio', 'image', 'gallery', 'quote'
		switch ($energy_post_format):
			case 'video':
				energy_icon('video');
				break;
			case 'audio':
				energy_icon('volume-high');
				break;
			case 'image':
				//energy_icon('image');
				break;
			case 'gallery':
				energy_icon('camera');
				break;
			case 'quote':
				energy_icon('format-quote-close');
				break;

			default:
		endswitch;
	}
endif;


if (!function_exists('energy_the_author')) :
	/*
		** Prints author HTML with with link on author archive.
		 */
	function energy_the_author()
	{

		//options
		//single post
		if (is_singular()) {

			$energy_show_author   = energy_option('blog_single_show_author', true);
			$energy_author_avatar = energy_option('blog_single_show_author_avatar', '');
			$energy_author_word   = energy_option('blog_single_before_author_word', '');
			$energy_show_icons    = !energy_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$energy_show_author   = energy_option('blog_show_author', true);
			$energy_author_avatar = energy_option('blog_show_author_avatar', '');
			$energy_author_word   = energy_option('blog_before_author_word', '');
			$energy_show_icons    = !energy_option('blog_hide_meta_icons', false);
		}

		if (!empty($energy_show_author)) :
			//author-wrapper
			if (!empty($energy_author_avatar)) :
				echo '<span class="author-avatar">';
				$energy_author_id        = get_the_author_meta('ID');
				$energy_custom_image_url = get_the_author_meta('custom_profile_image', $energy_author_id);
				if (!empty($energy_custom_image_url)) {
					echo '<img src="' . esc_url($energy_custom_image_url) . '" alt="' . esc_attr(get_the_author_meta('display_name', $energy_author_id)) . '">';
				} else {
					echo get_avatar($energy_author_id, 60);
				}
				echo '</span><!-- .author-avatar-->';
			endif; //$energy_author_avatar
		?>
			<span class="entry-author-wrap icon-inline">
				<?php
				//icon
				if (!empty($energy_show_icons)) {
					energy_icon('user');
				}
				//word
				if (!empty($energy_author_word)) :
				?>
					<span class="entry-author-word meta-word">
						<?php echo esc_html($energy_author_word); ?>
					</span>
					<!--.entry-author-word-->
				<?php
				endif;
				//value
				?>
				<span class="vcard author" itemtype="https://schema.org/Person" itemscope="itemscope" itemprop="author">
					<?php
					the_author_posts_link();
					?>
				</span><!-- .author -->
			</span>
			<!--.entry-author-wrap-->
			<!-- publisher -->
			<div class="hidden" itemprop="publisher" itemtype="http://schema.org/Organization" itemscope="itemscope">
				<span itemprop="name"><?php the_author(); ?></span>
				<?php
				$energy_custom_logo = energy_option('custom_logo');
				if (!empty($energy_custom_logo)) :
				?>
					<span itemprop="logo" itemscope="itemscope" itemtype="https://schema.org/ImageObject">
						<?php
						$energy_custom_logo_metadata = !empty($energy_custom_logo) ? wp_get_attachment_metadata($energy_custom_logo) : array(
							'width'  => '0',
							'height' => '0',
						);
						echo wp_get_attachment_image($energy_custom_logo, 'full');
						?>
						<meta itemprop="url" content="<?php echo esc_url(home_url('/')); ?>" />
						<meta itemprop="width" content="<?php echo esc_attr($energy_custom_logo_metadata['width']); ?>" />
						<meta itemprop="height" content="<?php echo esc_attr($energy_custom_logo_metadata['height']); ?>" />
					</span>
				<?php endif; //custom_logo 
				?>
			</div>
			<!-- publisher -->
		<?php
		endif; //author
	}
endif; //energy_the_author

add_filter('the_author_posts_link', 'energy_the_author_link_itemprop');
if (!function_exists('energy_the_author_link_itemprop')) :
	/**
	 * Add 'itemprop' attribute to the author link.
	 */
	function energy_the_author_link_itemprop($energy_link)
	{
		$energy_link = str_replace('rel="author">', 'rel="author" itemprop="url"><span itemprop="name">', $energy_link);
		$energy_link = str_replace('</a>', '</span></a>', $energy_link);

		return $energy_link;
	}
endif;

if (!function_exists('energy_the_date')) :
	/**
	 * Prints date HTML with the post link on blog.
	 */
	function energy_the_date()
	{

		//options
		//single post
		if (is_singular()) {

			$energy_show_date  = energy_option('blog_single_show_date', true);
			$energy_date_word  = energy_option('blog_single_before_date_word', '');
			$energy_show_icons = !energy_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$energy_show_date  = energy_option('blog_show_date', true);
			$energy_date_word  = energy_option('blog_before_date_word', '');
			$energy_show_icons = !energy_option('blog_hide_meta_icons', false);
		}

		if (!empty($energy_show_date)) :
			//date-wrapper
		?>
			<span class="entry-date-wrap icon-inline">
				<?php

				//icon
				if (!empty($energy_show_icons)) {
					energy_icon('clock-outline');
				}

				//word
				if (!empty($energy_date_word)) :
				?>
					<span class="date-word meta-word">
						<?php echo esc_html($energy_date_word); ?>
					</span>
					<!--.date-word-->
				<?php
				endif;
				//value
				//link date to post on archive
				if (!is_singular()) :
				?>
					<a href="<?php echo esc_url(get_permalink()); ?>" rel="bookmark" itemprop="mainEntityOfPage">
					<?php endif; //is_singular 
					?>
					<span itemprop="datePublished"><?php the_time(get_option('date_format')); ?></span>
					<span class="hidden" itemprop="dateModified"><?php the_modified_time(get_option('date_format')); ?></span>
					<?php if (!is_singular()) : ?>
					</a>
				<?php endif; //is_singular 
				?>
			</span>
			<!--.entry-date-wrap-->
			<?php
		endif; //date
	}
endif; //energy_the_date

if (!function_exists('energy_the_categories')) :
	/**
	 * Prints categories HTML for the current post.
	 */

	function energy_the_categories()
	{

		//options
		//single post
		if (is_singular()) {

			$energy_show_categories = energy_option('blog_single_show_categories', true);
			$energy_categories_word = energy_option('blog_single_before_categories_word', '');
			$energy_show_icons      = !energy_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$energy_show_categories = energy_option('blog_show_categories', true);
			$energy_categories_word = energy_option('blog_before_categories_word', '');
			$energy_show_icons      = !energy_option('blog_hide_meta_icons', false);
		}

		if (!empty($energy_show_categories)) :
			$energy_c = wp_get_post_categories(get_the_ID());

			//only if categories exists
			if (!empty($energy_c)) :

				//categories-wrapper
			?>
				<span class="entry-categories-wrap icon-inline">
					<?php

					//icon
					if (!empty($energy_show_icons)) {
						energy_icon('folder-outline');
					}

					//word
					if (!empty($energy_categories_word)) :
					?>
						<span class="categories-word meta-word">
							<?php echo esc_html($energy_categories_word); ?>
						</span>
						<!--.categories-word-->
					<?php
					endif;

					//value
					?>
					<span class="categories-list">
						<?php
						echo wp_kses_post(get_the_category_list('<span class="entry-categories-separator"> </span>'));
						?>
					</span>
					<!--.categories-list-->
				</span>
				<!--.entry-categories-wrap-->
			<?php
			endif; //$energy_c
		endif; //categories
	}
endif; //energy_the_categories

if (!function_exists('energy_the_tags')) :
	/**
	 * Prints tags HTML for the current post.
	 */
	function energy_the_tags()
	{

		//options
		//single post
		if (is_singular()) {

			$energy_show_tags  = energy_option('blog_single_show_tags', true);
			$energy_tags_word  = energy_option('blog_single_before_tags_word', '');
			$energy_show_icons = !energy_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$energy_show_tags  = energy_option('blog_show_tags', true);
			$energy_tags_word  = energy_option('blog_before_tags_word', '');
			$energy_show_icons = !energy_option('blog_hide_meta_icons', false);
		}

		if (!empty($energy_show_tags)) :

			$energy_t = wp_get_post_tags(get_the_ID());

			//only if tags exists
			if (!empty($energy_t)) :

				//tags-wrapper
			?>
				<span class="entry-tags-wrap icon-inline">
					<?php

					//icon
					if (!empty($energy_show_icons)) {
						energy_icon('tag');
					}

					//word
					if (!empty($energy_tags_word)) :
					?>
						<span class="tags-word meta-word">
							<?php echo esc_html($energy_tags_word); ?>
						</span>
						<!--.tags-word-->
					<?php
					endif; //tags_word

					//value
					?>
					<span class="tags-list">
						<?php
						echo wp_kses_post(get_the_tag_list('<span class="entry-tags">', '<span class="entry-tags-separator"> </span>', '</span>'));
						?>
					</span>
					<!--.tags-list-->
				</span>
				<!--.entry-tags-wrap-->
				<?php
			endif; //$energy_t
		endif; //tags
	}
endif; //energy_the_tags

if (!function_exists('energy_comment_count')) :
	/**
	 * Prints HTML with the comment count for the current post.
	 */
	function energy_comment_count()
	{

		//options
		//single post
		if (is_singular()) {

			$energy_show_comments = energy_option('blog_single_show_comments_link', 'number');
			$energy_show_icons    = !energy_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$energy_show_comments = energy_option('blog_show_comments_link', 'number');
			$energy_show_icons    = !energy_option('blog_hide_meta_icons', false);
		}

		if (!post_password_required() && (comments_open() || get_comments_number()) && $energy_show_comments) :
			switch ($energy_show_comments):
				case 'number':
				?>
					<span class="comments-link icon-inline">
						<?php
						if (!empty($energy_show_icons)) {
							energy_icon('comment-outline');
						}
						$comments_count = get_comments_number();
						comments_popup_link($comments_count, $comments_count, $comments_count);
						?>
					</span><!-- .comments-link -->
				<?php
					break;
					//text
				default:
				?>
					<span class="comments-link icon-inline">
						<?php

						if (!empty($energy_show_icons)) {
							energy_icon('comment-outline');
						}
						?>

						<?php

						comments_popup_link(
							sprintf(
								wp_kses(
									/* translators: %s: Name of current post. Only visible to screen readers. */
									__(' Leave a comment<span class="screen-reader-text"> on %s</span>', 'energy'),
									array(
										'span' => array(
											'class' => array(),
										),
									)
								),
								get_the_title()
							)
						);
						?>
					</span><!-- .comments-link -->
				<?php
			endswitch;
		endif; //show_comments
	}
endif;

if (!function_exists('energy_set_post_likes')) :
	/**
	 * Likes incrementor
	 *
	 * @param int $postID ID of the post.
	 *
	 * @return bool No success if cookies are disabled
	 */
	function energy_set_post_likes($postID)
	{
		if (empty($_COOKIE["$postID"])) {

			$count_key = 'energy_post_likes_count';
			$count     = get_post_meta($postID, $count_key, true);
			if ($count == '') {
				$count = 0;
				delete_post_meta($postID, $count_key);
				add_post_meta($postID, $count_key, '1');
			} else {
				$count++;
				update_post_meta($postID, $count_key, $count);
			}
			setcookie("$postID", "voted", strtotime('+1 day'), COOKIEPATH, COOKIE_DOMAIN, false); // 86400 = 1 day
			return true;
		}

		return false;
	} //energy_set_post_likes()
endif;

if (!function_exists('energy_get_post_likes')) :
	/**
	 * Get likes value
	 *
	 * @param int $postID ID of the post.
	 */
	function energy_get_post_likes($postID)
	{
		$count_key = 'energy_post_likes_count';
		$count     = get_post_meta($postID, $count_key, true);
		if ($count == '') {
			delete_post_meta($postID, $count_key);
			add_post_meta($postID, $count_key, '0');

			return '0';
		}

		return $count;
	} //energy_get_post_likes()
endif;

if (!function_exists('energy_print_post_likes')) :
	/**
	 * Get likes value
	 *
	 * @param int $count of likes of the post.
	 */
	function energy_print_post_likes($count)
	{
		$html = '';
		if (!$count) {
			$html = '<span class="item-likes-count">0</span> <span class="item-likes-word">' . esc_html__('Likes', 'energy') . '</span>';
		}

		if ($count == 1) {
			$html = '<span class="item-likes-count">1</span> <span class="item-likes-word">' . esc_html__('Like', 'energy') . '</span>';
		}

		if ($count > 1) {
			$html = '<span class="item-likes-count">' . $count . '</span> <span class="item-likes-word">' . esc_html__('Likes', 'energy') . '</span>';
		}

		return $html;
	} //energy_print_post_likes()
endif;

if (!function_exists('energy_post_likes_scripts')) :
	// Add the JS
	function energy_post_likes_scripts()
	{
		wp_enqueue_script('post-likes', get_template_directory_uri() . '/assets/js/mod-post-likes.js', array('jquery'), '1.0.0', true);
		wp_localize_script('post-likes', 'MyAjax', array(
			// URL to wp-admin/admin-ajax.php to process the request
			'ajaxurl'  => admin_url('admin-ajax.php'),
			// generate a nonce with a unique ID "myajax-post-comment-nonce"
			// so that you can check it later when an AJAX request is sent
			'security' => wp_create_nonce('increment-post-likes') //,
			//'post_id' => get_the_ID()
		));
	} //energy_post_likes_scripts()
endif;
add_action('wp_enqueue_scripts', 'energy_post_likes_scripts');

if (!function_exists('energy_inc_post_like_callback')) :
	// The function that handles the AJAX request
	function energy_inc_post_like_callback()
	{
		check_ajax_referer('increment-post-likes', 'security');
		$pID = intval($_POST['pID']);
		energy_set_post_likes($pID);
		echo energy_print_post_likes(energy_get_post_likes($pID));

		die(); // this is required to return a proper result
	} //energy_inc_post_like_callback()
endif;
add_action('wp_ajax_add_like', 'energy_inc_post_like_callback');
add_action('wp_ajax_nopriv_add_like', 'energy_inc_post_like_callback');

if (!function_exists('energy_post_like_button')) :
	/**
	 * Print like button
	 */
	function energy_post_like_button($postID)
	{
		$output = '';
		if (empty($_COOKIE["$postID"])) {
			$output = '<span data-id="' . $postID . '"><a href="" class="like_button highlight like_active_button"><i class="fa fa-heart-o" aria-hidden="true"></i></a></span>';
		} else {
			$output = '<span data-id="' . $postID . '"><span class="like_button highlight"><i class="fa fa-check" aria-hidden="true"></i></span></span>';
		}
		echo apply_filters('energy_like_button', $output);
	} //energy_post_like_button()
endif;
add_action('energy_post_meta', 'energy_post_like_button', 10, 1);

if (!function_exists('energy_post_like_count')) :
	/**
	 * Print like counter value
	 */
	function energy_post_like_count($postID)
	{
		echo apply_filters('energy_like_count', '<span class="item-likes votes_count_' . $postID . '">' . energy_print_post_likes(energy_get_post_likes($postID)) . '</span>');
	} //energy_post_like_count()
endif;
add_action('energy_post_meta', 'energy_post_like_count', 20, 1);

if (!function_exists('energy_the_likes')) :
	/**
	 * Prints date HTML with the post link on blog.
	 */
	function energy_the_likes()
	{

		//options
		//single post
		if (is_singular()) {

			$blog_show_likes  = energy_option('blog_single_show_likes', 'number');
			$blog_before_likes_word  = energy_option('blog_single_before_likes_word', '');

			//blog loop
		} else {

			$blog_show_likes  = energy_option('blog_show_likes', 'number');
			$blog_before_likes_word  = energy_option('blog_before_likes_word', '');
		}

		if (!empty($blog_show_likes)) :
			//likes-wrapper
			switch ($blog_show_likes):
				case 'number':
				?>
					<span class="entry-like-wrap icon-inline number-only">
						<?php

						//icon
						energy_post_like_button(get_the_ID());

						//word
						if (!empty($blog_before_likes_word)) :
						?>
							<span class="date-word meta-word">
								<?php echo esc_html($blog_before_likes_word); ?>
							</span>
							<!--.likes-word-->
						<?php
						endif;
						energy_post_like_count(get_the_ID());
						?>
					</span>
				<?php
					break;
					//text
				default:
				?>
					<span class="entry-view-wrap icon-inline">
						<?php

						//icon
						energy_post_like_button(get_the_ID());

						//word
						if (!empty($blog_before_likes_word)) :
						?>
							<span class="date-word meta-word">
								<?php echo esc_html($blog_before_likes_word); ?>
							</span>
							<!--.likes-word-->
						<?php
						endif;
						energy_post_like_count(get_the_ID());
						?>
					</span>
			<?php
			endswitch;
		endif; //likes
	}
endif; //energy_show_likes

if (!function_exists('energy_the_view')) :
	/**
	 * Prints date HTML with the post link on blog.
	 */
	function energy_the_view()
	{

		//options
		//single post
		if (is_singular()) {

			$blog_show_views  = energy_option('blog_single_show_views', true);
			$blog_before_views_word  = energy_option('blog_single_before_views_word', '');
			$energy_show_icons = !energy_option('blog_single_hide_meta_icons', false);

			//blog loop
		} else {

			$blog_show_views  = energy_option('blog_show_views', true);
			$blog_before_views_word  = energy_option('blog_before_views_word', '');
			$energy_show_icons = !energy_option('blog_hide_meta_icons', false);
		}

		if (!empty($blog_show_views) && function_exists('energy_show_post_views_count')) :
			//date-wrapper
			?>
			<span class="entry-view-wrap icon-inline">
				<?php

				//icon
				if (!empty($energy_show_icons)) {
					energy_icon('people');
				}

				//word
				if (!empty($blog_before_views_word)) :
				?>
					<span class="date-word meta-word">
						<?php echo esc_html($blog_before_views_word); ?>
					</span>
					<!--.date-word-->
			<?php
				endif;
				energy_show_post_views_count();
				echo '</span>';
			endif; //date
		}
	endif; //energy_the_date

	if (!function_exists('energy_entry_meta')) :
		/**
		 * Prints HTML with the comment count for the current post.
		 */

		function energy_entry_meta($energy_show_author = true, $energy_show_date = true, $energy_show_categories = true, $energy_show_tags = true, $energy_show_comments = true, $energy_show_likes = true, $energy_the_view = true)
		{
			if (!empty($energy_show_author)) :
				energy_the_author();
			endif; //author

			if (!empty($energy_show_date)) :
				energy_the_date();
			endif; //date

			if (!empty($energy_show_categories)) :
				energy_the_categories();
			endif; //categories

			if (!empty($energy_show_tags)) :
				energy_the_tags();
			endif; //tags

			if (!empty($energy_show_comments)) :
				energy_comment_count();
			endif; //comments

			if (!empty($energy_show_likes)) :
				energy_the_likes();
			endif; //likes

			if (!empty($energy_the_view)) :
				energy_the_view();
			endif; //view
		}
	endif;


	if (!function_exists('energy_post_nav')) :
		/**
		 * Display navigation to next/previous post when applicable.
		 */
		function energy_post_nav()
		{

			$energy_blog_single_post_nav = energy_option('blog_single_post_nav', '');

			if (empty($energy_blog_single_post_nav)) {
				return;
			}

			// Don't print empty markup if there's nowhere to navigate.
			$energy_previous = (is_attachment()) ? get_post(get_post()->post_parent) : get_adjacent_post(false, '', true);
			$energy_next     = get_adjacent_post(false, '', false);

			if (!$energy_next && !$energy_previous) {
				return;
			}

			$energy_word_prev = energy_option('blog_single_post_nav_word_prev', esc_html__('Prev', 'energy'));
			$energy_word_next = energy_option('blog_single_post_nav_word_next', esc_html__('Next', 'energy'));
			?>
			<nav class="post-nav post-nav-layout-<?php echo esc_attr($energy_blog_single_post_nav); ?>">
				<?php

				if (is_attachment() && 'attachment' === $energy_previous->post_type) {
					return;
				}

				if ($energy_previous && has_post_thumbnail($energy_previous->ID)) {
					$energy_prevthumb = wp_get_attachment_image_src(get_post_thumbnail_id($energy_previous->ID), 'post-thumbnail');
					if ($energy_prevthumb) {
						$energy_prevthumb_sm         = wp_get_attachment_image_src(get_post_thumbnail_id($energy_previous->ID), 'thumbnail');
						$energy_prev_thumbnail_style = ' style="background-image: url(' . esc_url($energy_prevthumb[0]) . '); "';
						$energy_prev_thumbnail_class = 'has-image background-cover background-overlay';
						$energy_prev_thumbnail_img   = '<span class="post-nav-thumb"><img src="' . esc_url($energy_prevthumb_sm[0]) . '" alt="' . $energy_previous->post_title . '"></span>';
					} else {
						$energy_prev_thumbnail_style = '';
						$energy_prev_thumbnail_class = 'no-image';
						$energy_prev_thumbnail_img   = '';
					}
				} else {
					$energy_prev_thumbnail_style = '';
					$energy_prev_thumbnail_class = 'no-image';
					$energy_prev_thumbnail_img   = '';
				}

				if ($energy_next && has_post_thumbnail($energy_next->ID)) {
					$energy_nextthumb = wp_get_attachment_image_src(get_post_thumbnail_id($energy_next->ID), 'post-thumbnail');
					if ($energy_nextthumb) {
						$energy_nextthumb_sm         = wp_get_attachment_image_src(get_post_thumbnail_id($energy_next->ID), 'thumbnail');
						$energy_next_thumbnail_style = ' style="background-image: url(' . esc_url($energy_nextthumb[0]) . '); "';
						$energy_next_thumbnail_class = 'has-image background-cover background-overlay';
						$energy_next_thumbnail_img   = '<span class="post-nav-thumb"><img src="' . esc_url($energy_nextthumb_sm[0]) . '" alt="' . $energy_next->post_title . '"></span>';
					} else {
						$energy_next_thumbnail_style = '';
						$energy_next_thumbnail_class = 'no-image';
						$energy_next_thumbnail_img   = '';
					}
				} else {
					$energy_next_thumbnail_style = '';
					$energy_next_thumbnail_class = 'no-image';
					$energy_next_thumbnail_img   = '';
				}

				//layouts
				switch ($energy_blog_single_post_nav):
					case 'bg':
						echo '<div>';
						previous_post_link(
							'%link',
							'<div class="post-nav-item bg-item prev-item ' . esc_attr($energy_prev_thumbnail_class) . '"' . $energy_prev_thumbnail_style . '>
							<span class="post-nav-words-wrap">
								<span class="post-nav-word highlight">' . esc_html($energy_word_prev) . '</span>
								<h4 class="post-nav-title">%title</h4>
							</span>
						</div>',
							false,
							''
						);
						echo '</div>';

						echo '<div>';
						next_post_link(
							'%link',
							'<div class="post-nav-item bg-item next-item ' . esc_attr($energy_next_thumbnail_class) . '"' . $energy_next_thumbnail_style . '>
							<span class="post-nav-words-wrap">
								<span class="post-nav-word highlight">' . esc_html($energy_word_next) . '</span>
								<h4 class="post-nav-title">%title</h4>
							</span>
						</div>',
							false,
							''
						);
						echo '</div>';
						break;
					case 'thumbnail':
						echo '<div>';
						previous_post_link(
							'%link',
							'<div class="post-nav-item prev-item">
							<span class="post-nav-arrow">' . energy_icon('chevron-left', true) . '</span>
							' . $energy_prev_thumbnail_img . '
							<span class="post-nav-words-wrap">
								<span class="post-nav-word">' . esc_html($energy_word_prev) . '</span>
								<span class="post-nav-title">%title</span>
							</span>
						</div>',
							false,
							''
						);
						echo '</div>';

						echo '<div>';
						next_post_link(
							'%link',
							'<div class="post-nav-item next-item">
							<span class="post-nav-words-wrap">
								<span class="post-nav-word">' . esc_html($energy_word_next) . '</span> 
								<span class="post-nav-title">%title</span>
							</span>
							' . $energy_next_thumbnail_img . '
							<span class="post-nav-arrow">' . energy_icon('chevron-right', true) . '</span>
						</div>',
							false,
							''
						);
						echo '</div>';
						break;
					case 'arrow':
						echo '<div>';
						previous_post_link(
							'%link',
							'<div class="post-nav-item prev-item">
								<span class="post-nav-word small-text">' . esc_html($energy_word_prev) . '</span>
							</div>',
							false,
							''
						);
						echo '</div>';

						echo '<div>';
						next_post_link(
							'%link',
							'<div class="post-nav-item next-item">
								<span class="post-nav-word small-text">' . esc_html($energy_word_next) . '</span> 
							</div>',
							false,
							''
						);
						echo '</div>';
						break;
						//title
					default:
						echo '<div>';
						previous_post_link(
							'%link',
							'<div class="post-nav-item prev-item">
							<span class="post-nav-words-wrap small-text">
								<span class="post-nav-word">' . esc_html($energy_word_prev) . '</span>
								<span class="post-nav-title">%title</span>
							</span>
						</div>',
							false,
							''
						);
						echo '</div>';

						echo '<div>';
						next_post_link(
							'%link',
							'<div class="post-nav-item next-item">
							<span class="post-nav-words-wrap small-text">
								<span class="post-nav-word">' . esc_html($energy_word_next) . '</span> 
								<span class="post-nav-title">%title</span>
							</span>
						</div>',
							false,
							''
						);
						echo '</div>';
				endswitch;

				?>
			</nav><!-- .navigation -->
			<?php
		} //energy_post_nav
	endif;


	if (!function_exists('energy_section_background_image_array')) :
		/**
		 * Get array of section attributes to display background image.
		 */
		function energy_section_background_image_array($energy_section, $energy_empty_image = false)
		{

			//processing title section background for simple single post 'title-section-image' layout
			if (is_single() && 'title' === $energy_section) :
				if (energy_get_post_layout() === 'title-section-image') :
					//if has post thumbnail
					if (!post_password_required() && !is_attachment() && has_post_thumbnail()) {
						return array(
							'url'   => get_the_post_thumbnail_url(get_the_ID(), 'full'),
							'class' => 'i post-thumbnail-background background-cover cover-center background-fixed background-overlay overlay-dark',
						);
					}
				endif;
			endif; //is_single

			//for page with feature image - override default header_image
			if ('header_image' === $energy_section) {
				$energy_image = get_header_image();
				//for page with feature image - override default image
				if (is_page()) {
					if (has_post_thumbnail()) {
						$energy_image = get_the_post_thumbnail_url();
					}
				}
			} else {
				$energy_image = energy_option($energy_section . '_background_image', '');
				// override title background if page featured image is set
				if ('title' === $energy_section && $energy_image) {
					//for page with feature image - override default image
					if (is_page()) {
						if (has_post_thumbnail()) {
							$energy_image = get_the_post_thumbnail_url();
						}
					}
				}
			}

			$energy_return = array(
				'url'   => $energy_image,
				'class' => '',
			);

			if (empty($energy_image) && empty($energy_empty_image)) {
				return $energy_return;
			}

			$energy_cover   = energy_option($energy_section . '_background_image_cover', '');
			$energy_fixed   = energy_option($energy_section . '_background_image_fixed', '');
			$energy_overlay = energy_option($energy_section . '_background_image_overlay', '');

			if (!empty($energy_cover)) {
				$energy_return['class'] .= 'background-cover cover-center';
			}

			if (!empty($energy_fixed)) {
				$energy_return['class'] .= ' background-fixed';
			}

			if (!empty($energy_overlay)) {
				$energy_return['class'] .= ' background-overlay ' . $energy_overlay;
			}

			return $energy_return;
		}
	endif;

	/////////////
	//Read More//
	/////////////

	// Read more markup inside link for excertp and the_content
	if (!function_exists('energy_read_more_inside_link_markup')) :
		function energy_read_more_inside_link_markup($energy_read_more_text = '')
		{

			if (empty($energy_read_more_text)) {
				$energy_read_more_text = energy_option('blog_read_more_text', '');
			}

			if (empty($energy_read_more_text)) {
				return '';
			}

			return sprintf(
				wp_kses(
					$energy_read_more_text . '<span class="screen-reader-text"> "%s"</span>',
					array(
						'span' => array(
							'class' => array(),
						),
					)
				),
				get_the_title()
			);
		}
	endif;

	//generated excerpt ending...
	if (!function_exists('energy_excerpt_more')) :
		function energy_excerpt_more($energy_more)
		{
			return '<span class="more-dots">...</span>';
		}
	endif;
	add_filter('excerpt_more', 'energy_excerpt_more', 21);

	//read more for excerpt
	if (!function_exists('energy_read_more_markup_excerpt')) :
		function energy_read_more_markup_excerpt()
		{
			global $post;
			$energy_markup = ' <span class="more-tag"><a class="more-link" href="' .
				esc_url(get_permalink($post->ID)) . '">' .
				energy_read_more_inside_link_markup() .
				'</a></span><!-- .more-tag -->';

			return $energy_markup;
		}
	endif;

	//putting read more text inside excerpt if text is not empty
	if (!function_exists('energy_read_more_in_excerpt')) :
		function energy_read_more_in_excerpt($energy_excerpt)
		{

			$energy_read_more_text = energy_option('blog_read_more_text', '');

			if (empty($energy_read_more_text)) {
				return $energy_excerpt;
			}

			$energy_excerpt = str_replace('</p>', energy_read_more_markup_excerpt($energy_read_more_text) . '</p>', $energy_excerpt);

			return $energy_excerpt;
		}
	endif;
	add_filter('the_excerpt', 'energy_read_more_in_excerpt', 21);

	//Filter the except length
	if (!function_exists('energy_excerpt_custom_length')) :
		function energy_excerpt_custom_length($energy_length)
		{

			return absint(energy_option('blog_excerpt_length', 40));
		}
	endif;
	add_filter('excerpt_length', 'energy_excerpt_custom_length', 999);

	//home page intro teasers
	if (!function_exists('energy_get_intro_teasers')) :
		function energy_get_intro_teasers()
		{

			$energy_teasers = array();

			for ($energy_i = 1; $energy_i < 5; $energy_i++) {
				/*
			reeatable options:
				intro_teaser_image_
				intro_teaser_title_
				intro_teaser_text_
				intro_teaser_link_
				intro_teaser_button_text_
			*/
				$energy_teasers[$energy_i] = array_filter(
					array(
						'image'  => energy_option('intro_teaser_image_' . $energy_i, ''),
						'title'  => energy_option('intro_teaser_title_' . $energy_i, ''),
						'text'   => energy_option('intro_teaser_text_' . $energy_i, ''),
						'link'   => energy_option('intro_teaser_link_' . $energy_i, ''),
						'button' => energy_option('intro_teaser_button_text_' . $energy_i, ''),
					)
				);
			}

			return array_filter($energy_teasers);
		}
	endif;

	//related posts
	if (!function_exists('energy_related_posts')) :
		function energy_related_posts($energy_id)
		{

			$energy_layout = energy_option('blog_single_related_posts', '');
			if (empty($energy_layout)) {
				return;
			}
			$energy_tags   = wp_get_post_tags($energy_id, array('fields' => 'ids'));
			if (!empty($energy_tags)) :
				//list
				//list-thumbnails
				//grid
				//num of posts
				$energy_posts_number = absint(energy_option('blog_single_related_posts_number', 3));
				if (empty($energy_posts_number)) {
					$energy_posts_number = 3;
				}

				$energy_args  = array(
					'tag__in'        => $energy_tags,
					'post__not_in'   => array($energy_id),
					'posts_per_page' => $energy_posts_number,
				);
				$energy_query = new WP_Query($energy_args);
				if ($energy_query->have_posts()) :
					$energy_related_title = energy_option('blog_single_related_posts_title', esc_html__('Related Posts', 'energy'));
			?>
					<div class="related-posts content-padding with_shadow with-bottom-border">
						<?php if (!empty($energy_related_title)) : ?>
							<h4 class="related-posts-heading"><?php echo wp_kses_post($energy_related_title); ?></h4>
							<?php
						endif; //related_title
						switch ($energy_layout):
							case 'grid':
								switch ($energy_posts_number):
									case 3:
										$energy_wrapper_class = 'layout-cols-3';
										break;
									case 4:
										$energy_wrapper_class = 'layout-cols-4';
										break;
									default:
										$energy_wrapper_class = '';
								endswitch;
								if ($energy_query->post_count < 3) {
									$energy_wrapper_class = 'layout-cols-' . $energy_query->post_count;
								}
							?>
								<div class="layout-gap-30">
									<div class="grid-wrapper <?php echo esc_attr($energy_wrapper_class); ?>">
										<?php
										while ($energy_query->have_posts()) :
											$energy_query->the_post();
										?>
											<div class="grid-item post">
												<article <?php post_class('vertical-item'); ?>>
													<?php if (has_post_thumbnail()) : ?>
														<figure class="post-thumbnail">
															<a href="<?php the_permalink(); ?>">
																<?php the_post_thumbnail('energy-default-post'); ?>
															</a>
														</figure>
													<?php endif; ?>
													<div class="item-content">
														<h5 class="entry-title">
															<a rel="bookmark" href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
														</h5>
														<div class="entry-meta post-meta small-text">
															<div class="meta-left"><?php energy_entry_meta(false, true, false, false, false, false, false); ?></div>
														</div>
													</div>
												</article>
											</div>
										<?php endwhile; ?>
									</div><!-- .grid-wrapper -->
								</div><!-- .layout-gap-* -->
							<?php
								break;

							case 'list-thumbnails':
							?>
								<ul class="posts-list">
									<?php
									while ($energy_query->have_posts()) :
										$energy_query->the_post();
									?>
										<li class="list-has-post-thumbnail">
											<?php if (has_post_thumbnail()) : ?>
												<a class="posts-list-thumbnail" href="<?php the_permalink(); ?>">
													<?php the_post_thumbnail('thumbnail'); ?>
												</a>
											<?php endif; ?>
											<div class="item-content">
												<h4 class="post-title">
													<a rel="bookmark" href="<?php the_permalink(); ?>"><?php the_title(); ?></a>
												</h4>
												<div class="icon-inline-wrap">
													<span class="icon-inline post-date">
														<?php energy_icon('clock-outline'); ?>
														<span><?php echo get_the_date('', get_the_ID()); ?></span>
													</span>
												</div>
											</div>
										</li>
									<?php endwhile; ?>
								</ul>
							<?php
								break;

							default:
							?>
								<ul class="list-styled">
									<?php
									while ($energy_query->have_posts()) :
										$energy_query->the_post();
									?>
										<li>
											<h6>
												<a href="<?php the_permalink(); ?>" rel="bookmark"><?php the_title(); ?></a>
											</h6>
										</li>
									<?php endwhile; ?>
								</ul>
						<?php
						endswitch; //layout
						?>
					</div><!-- .related-posts -->
			<?php
				endif; //have_posts
				wp_reset_postdata();
			endif; //tags
		}
	endif;

	//get any widget HTML markup
	if (!function_exists('energy_get_the_widget')) :
		function energy_get_the_widget($energy_widget_class, $energy_instance = array())
		{

			if (!class_exists($energy_widget_class)) {
				return '';
			}
			//same as in inc/setup.php file
			$energy_args = array(
				'before_title' => '<h3 class="widget-title"><span>',
				'after_title'  => '</span></h3>',
			);

			ob_start();

			the_widget($energy_widget_class, $energy_instance, $energy_args);

			return ob_get_clean();
		}
	endif;

	//count widgets in sidebar
	if (!function_exists('energy_get_sidebar_widgets_count')) :
		function energy_get_sidebar_widgets_count($sidebar_id)
		{
			$widgets = get_option('sidebars_widgets');

			return count($widgets[$sidebar_id]);
		}
	endif;

	//detect shop - handy for sidebar and breadcrumbs
	if (!function_exists('energy_is_shop')) :
		function energy_is_shop()
		{
			$energy_return = false;
			if (function_exists('is_woocommerce')) {
				if (is_woocommerce() || is_cart() || is_checkout() || is_account_page()) {
					$energy_return = true;
				}
			}
			if (function_exists('yith_wcwl_is_wishlist_page')) {
				if (yith_wcwl_is_wishlist_page()) {
					$energy_return = true;
				}
			}

			return $energy_return;
		}
	endif;

	//echo breadcrumbs markup
	if (!function_exists('energy_breadcrumbs')) :
		function energy_breadcrumbs()
		{
			$energy_args              = array(
				'before' => '<nav class="breadcrumbs">',
				'after'  => '</nav>',
			);
			$energy_seo_options       = get_option('wpseo_titles');
			$energy_args['delimiter'] = !empty($energy_seo_options['breadcrumbs-sep']) ? $energy_seo_options['breadcrumbs-sep'] : '/';
			if (energy_is_shop()) :
				woocommerce_breadcrumb(
					array(
						'wrap_before' => $energy_args['before'] . '<span>',
						'wrap_after'  => '</span>' . $energy_args['after'],
						'before'      => '<span class="breadcrumbs_item">',
						'after'       => '</span>',
						'delimiter'   => ' ' . $energy_args['delimiter'] . ' ',
					)
				);
			elseif (function_exists('yoast_breadcrumb')) :
				yoast_breadcrumb('<nav class="breadcrumbs">', '</nav>');
			elseif (function_exists('rank_math_the_breadcrumbs')) :
				$args = array(
					'delimiter'   => '&nbsp;&#47;&nbsp;',
					'wrap_before' => '<nav class="breadcrumbs">',
					'wrap_after'  => '</nav>',
					'before'      => '',
					'after'       => '',
				);
				rank_math_the_breadcrumbs($args);
			endif;
		}
	endif;

	//check if breadcrumbs are enabled and plugins to show them are active
	if (!function_exists('energy_breadcrumbs_enabled')) :
		function energy_breadcrumbs_enabled()
		{
			$energy_return = energy_option('title_show_breadcrumbs', true);
			if (energy_is_shop() && $energy_return) {
				return $energy_return;
			} elseif (function_exists('yoast_breadcrumb') && $energy_return) {
				return $energy_return;
			} elseif (function_exists('rank_math_the_breadcrumbs') && $energy_return) {
				return $energy_return;
			} else {
				return false;
			}
		}
	endif;

	//copyright text - year
	if (!function_exists('energy_get_copyright_text')) :
		function energy_get_copyright_text($energy_text = '')
		{
			$energy_text = str_replace('[year]', '<span class="copyright-year">' . date('Y') . '</span>', $energy_text);
			return $energy_text;
		}
	endif;

	//detect is_front_page and not is paged
	if (!function_exists('energy_is_front_page')) :
		function energy_is_front_page()
		{
			return is_front_page();
		}
	endif;

	//detect for displaying title section
	if (!function_exists('energy_is_title_section_is_shown')) :
		function energy_is_title_section_is_shown()
		{
			if (is_page_template('page-templates/no-sidebar-no-title.php')) {
				return false;
			}
			$energy_show_title       = energy_option('title_show_title', '');
			$energy_show_search      = energy_option('title_show_search', '');
			$energy_show_breadcrumbs = energy_breadcrumbs_enabled();
			$energy_is_front_page    = energy_is_front_page();

			if (!empty($energy_is_front_page) && empty($energy_show_search)) {
				return false;
			}

			if (empty($energy_show_title) && empty($energy_show_breadcrumbs) && empty($energy_show_search)) {
				return false;
			} else {
				return true;
			}
		}
	endif;

	/**
	 *  Display info corner / date
	 */

	if (!function_exists('energy_corner_date')) :
		function energy_corner_date()
		{ ?>
			<div class="entry-meta-corner">
				<?php
				// Set up and print post meta information.
				printf(
					'<span class="date"><time class="entry-date" datetime="%1$s">%3$s<strong>%2$s</strong></time></span>',
					esc_attr(get_the_date('c')),
					esc_html(get_the_date('j')),
					esc_html(get_the_date('M'))
				); ?>
			</div>
		<?php
		}
	endif; // energy_corner_date

	// Event share likes views block
	if (!function_exists('function_event_views')) :
		function function_event_views($postID)
		{
			$count_key = 'energy_event_views_count';
			$count = get_post_meta($postID, $count_key, true);
			if ($count == '') {
				$count = 1;
				delete_post_meta($postID, $count_key);
				add_post_meta($postID, $count_key, '1');
			} else {
				$count++;
				update_post_meta($postID, $count_key, $count);
			}
		}
	endif;

	// The shortcode function
	function energy_share_likes_views_shortcode()
	{
		ob_start();
		?>
		<div class="entry-header">
			<div class="muted_background post-adds content-justify">
				<div class="content-justify">
					<?php
					energy_share_this();
					energy_post_like_button(get_the_ID());
					energy_post_like_count(get_the_ID());
					?>
				</div>
				<div>
					<span class="views-count gradient_bg">
						<span class="item-views-count">
							<?php
							if (get_post_meta(get_the_ID(), 'energy_event_views_count', true) == '') {
								echo '0';
							} else {
								echo get_post_meta(get_the_ID(), 'energy_event_views_count', true);
							};
							?>
						</span>
						<span class="item-views-word">Views</span>
					</span>
				</div>
			</div>
		</div>
	<?php
		return ob_get_clean();
	}

	// Register shortcode
	add_shortcode('share_likes_views_block', 'energy_share_likes_views_shortcode');

	if (!function_exists('energy_post_time')) :
		/**
		 * Prints HTML with meta time information for posts.
		 */
		function energy_post_time()
		{
			if (in_array(get_post_type(), array('post', 'attachment'))) {
				$time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';

				if (get_the_time('U') !== get_the_modified_time('U')) {
					$time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time><time class="updated" datetime="%3$s">%4$s</time>';
				}

				$time_string = sprintf(
					$time_string,
					esc_attr(get_the_date('c')),
					get_the_date('F j, Y'),
					esc_attr(get_the_modified_date('c')),
					get_the_modified_date('F j, Y - G:i')
				);

				printf(
					'<span class="posted-on"><span class="screen-reader-text">%1$s </span><a href="%2$s" rel="bookmark">%3$s</a></span>',
					_x('Posted on', 'Used before publish date.', 'energy'),
					esc_url(get_day_link(get_the_date('Y'), get_the_date('m'), get_the_date('d'))),
					$time_string
				);
			}
		}
	endif;
